<?php
session_start();
require 'conexion.php';

// Asegurarse de que el usuario esté autenticado y tenga el rol correcto (cod_rol = 8)
if (!isset($_SESSION['user_id']) || !isset($_SESSION['role']) || $_SESSION['role'] != 8) {
    $_SESSION['error'] = 'Acceso no autorizado. Por favor, inicia sesión con una cuenta de Jefe de Departamento válida.';
    header("Location: login.php");
    exit;
}

$user_id = $_SESSION['user_id'];
$nombre_usuario = $_SESSION['Nom'];

// Obtener el cod_unidad y el nombre_unidad del jefe
$sql_jefe_unidad = "SELECT a.cod_unidad, u.nombre_unidad 
                    FROM accounts a 
                    INNER JOIN unidad u ON a.cod_unidad = u.cod_unidad 
                    WHERE a.ID = ?";
$cod_unidad_jefe = null;
$nombre_unidad_jefe = null;
if ($stmt = $conn->prepare($sql_jefe_unidad)) {
    $stmt->bind_param('i', $user_id);
    $stmt->execute();
    $stmt->bind_result($cod_unidad_jefe, $nombre_unidad_jefe);
    $stmt->fetch();
    $stmt->close();
}

if (is_null($cod_unidad_jefe) || is_null($nombre_unidad_jefe)) {
    $_SESSION['error'] = 'No tienes una unidad asignada o la unidad no está registrada. Contacta al administrador.';
    header("Location: login.php");
    exit;
}

// Obtener el mes y año actual
$current_month = date('m');
$current_year = date('Y');

// ===== Estadísticas mes actual (Pendientes solo ciertos tipos) =====
$sql_pendientes = "SELECT COUNT(*) as total 
                   FROM solicitudes s 
                   JOIN accounts a ON s.id_usuario = a.ID 
                   WHERE s.estado = 'Pendiente' 
                   AND a.cod_unidad = ? 
                   AND MONTH(s.fecha_solicitud) = ? 
                   AND YEAR(s.fecha_solicitud) = ?
                   AND s.tipo_permiso IN ('Personal', 'Oficial', 'Vacaciones')";
                   
if ($stmt = $conn->prepare($sql_pendientes)) {
    $stmt->bind_param('sii', $cod_unidad_jefe, $current_month, $current_year);
    $stmt->execute();
    $stmt->bind_result($pendientes);
    $stmt->fetch();
    $stmt->close();
}


$sql_aprobadas_jefe = "SELECT COUNT(*) as total 
                       FROM solicitudes s 
                       JOIN accounts a ON s.id_usuario = a.ID 
                       WHERE s.estado = 'Aprobada por Jefe Inmediato' 
                       AND a.cod_unidad = ? 
                       AND MONTH(fecha_solicitud) = ? 
                       AND YEAR(fecha_solicitud) = ?";
if ($stmt = $conn->prepare($sql_aprobadas_jefe)) {
    $stmt->bind_param('sii', $cod_unidad_jefe, $current_month, $current_year);
    $stmt->execute();
    $stmt->bind_result($aprobadas_jefe);
    $stmt->fetch();
    $stmt->close();
}

$sql_aprobadas_rrhh = "SELECT COUNT(*) as total 
                       FROM solicitudes s 
                       JOIN accounts a ON s.id_usuario = a.ID 
                       WHERE s.estado = 'Aprobada por Recursos Humanos' 
                       AND a.cod_unidad = ? 
                       AND MONTH(fecha_solicitud) = ? 
                       AND YEAR(fecha_solicitud) = ?";
if ($stmt = $conn->prepare($sql_aprobadas_rrhh)) {
    $stmt->bind_param('sii', $cod_unidad_jefe, $current_month, $current_year);
    $stmt->execute();
    $stmt->bind_result($aprobadas_rrhh);
    $stmt->fetch();
    $stmt->close();
}

$sql_rechazadas_jefe = "SELECT COUNT(*) as total 
                        FROM solicitudes s 
                        JOIN accounts a ON s.id_usuario = a.ID 
                        WHERE s.estado = 'Rechazada por Jefe Inmediato' 
                        AND a.cod_unidad = ? 
                        AND MONTH(fecha_solicitud) = ? 
                        AND YEAR(fecha_solicitud) = ?";
if ($stmt = $conn->prepare($sql_rechazadas_jefe)) {
    $stmt->bind_param('sii', $cod_unidad_jefe, $current_month, $current_year);
    $stmt->execute();
    $stmt->bind_result($rechazadas_jefe);
    $stmt->fetch();
    $stmt->close();
}

$sql_rechazadas_rrhh = "SELECT COUNT(*) as total 
                        FROM solicitudes s 
                        JOIN accounts a ON s.id_usuario = a.ID 
                        WHERE s.estado = 'Rechazada por Recursos Humanos' 
                        AND a.cod_unidad = ? 
                        AND MONTH(fecha_solicitud) = ? 
                        AND YEAR(fecha_solicitud) = ?";
if ($stmt = $conn->prepare($sql_rechazadas_rrhh)) {
    $stmt->bind_param('sii', $cod_unidad_jefe, $current_month, $current_year);
    $stmt->execute();
    $stmt->bind_result($rechazadas_rrhh);
    $stmt->fetch();
    $stmt->close();
}

$sql_urgentes = "SELECT COUNT(*) as total 
                 FROM solicitudes s 
                 JOIN accounts a ON s.id_usuario = a.ID 
                 WHERE s.estado = 'Pendiente' 
                 AND a.cod_unidad = ? 
                 AND s.fecha_salida <= DATE_ADD(CURDATE(), INTERVAL 3 DAY)";
if ($stmt = $conn->prepare($sql_urgentes)) {
    $stmt->bind_param('s', $cod_unidad_jefe);
    $stmt->execute();
    $stmt->bind_result($urgentes);
    $stmt->fetch();
    $stmt->close();
}

// Lógica para solicitudes mensuales (SIN CAMBIOS)
$sql_monthly_requests = "SELECT COUNT(*) as total 
                         FROM solicitudes s 
                         JOIN accounts a ON s.id_usuario = a.ID 
                         WHERE a.cod_unidad = ? 
                         AND MONTH(s.fecha_solicitud) = ? 
                         AND YEAR(s.fecha_solicitud) = ?";
if ($stmt = $conn->prepare($sql_monthly_requests)) {
    $stmt->bind_param('sii', $cod_unidad_jefe, $current_month, $current_year);
    $stmt->execute();
    $stmt->bind_result($monthly_requests);
    $stmt->fetch();
    $stmt->close();
} else {
    $monthly_requests = 0;
}
$max_requests_per_month = 10;
$remaining_requests = $max_requests_per_month - $monthly_requests;

// Obtener solicitudes pendientes (SIN CAMBIOS)
$sql_solicitudes_pendientes = "SELECT s.id, s.tipo_permiso, s.fecha_solicitud, s.fecha_salida, s.descripcion_motivo, a.Nom, a.Ape 
                               FROM solicitudes s 
                               JOIN accounts a ON s.id_usuario = a.ID 
                               WHERE s.estado = 'Pendiente' 
                               AND a.cod_unidad = ?";
if ($stmt = $conn->prepare($sql_solicitudes_pendientes)) {
    $stmt->bind_param('s', $cod_unidad_jefe);
    $stmt->execute();
    $result_solicitudes_pendientes = $stmt->get_result();
    $stmt->close();
}

// Procesar acciones de aprobar/rechazar (SIN CAMBIOS)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && isset($_POST['solicitud_id'])) {
    $solicitud_id = (int)$_POST['solicitud_id'];
    $action = $_POST['action'];
    $estado = ($action === 'aprobar') ? 'Aprobada por Jefe Inmediato' : 'Rechazada por Jefe Inmediato';

    $sql_update = "UPDATE solicitudes SET estado = ?, id_admin = ? WHERE id = ?";
    if ($stmt = $conn->prepare($sql_update)) {
        $stmt->bind_param('sii', $estado, $user_id, $solicitud_id);
        $stmt->execute();
        $stmt->close();
        header("Location: dashboard_jefe.php");
        exit;
    }
}

// ======================= OBTENER DATOS PARA EL CALENDARIO (TODOS LOS ESTADOS) =======================
$selected_month = isset($_GET['month']) ? (int)$_GET['month'] : date('n');
$selected_year  = isset($_GET['year']) ? (int)$_GET['year'] : date('Y');

if ($selected_month < 1 || $selected_month > 12) $selected_month = date('n');
if ($selected_year  < 2000 || $selected_year  > 2100) $selected_year  = date('Y');

$days_in_month       = cal_days_in_month(CAL_GREGORIAN, $selected_month, $selected_year);
$first_day_of_month  = date('w', strtotime("$selected_year-$selected_month-01"));

$sql_cal = "SELECT 
                DATE(s.fecha_salida) AS fecha,
                s.id,
                s.estado,
                s.tipo_permiso,
                s.fecha_solicitud,
                s.fecha_salida,
                CONCAT(a.Nom,' ',a.Ape) AS funcionario
            FROM solicitudes s
            JOIN accounts a ON s.id_usuario = a.ID
            WHERE a.cod_unidad = ?
              AND YEAR(s.fecha_salida) = ?
              AND MONTH(s.fecha_salida) = ?
            ORDER BY s.fecha_salida, s.id";

$cal_por_dia = [];
if ($stmt = $conn->prepare($sql_cal)) {
    $stmt->bind_param('sii', $cod_unidad_jefe, $selected_year, $selected_month);
    $stmt->execute();
    $res = $stmt->get_result();
    while ($row = $res->fetch_assoc()) {
        $f = $row['fecha'];
        if (!isset($cal_por_dia[$f])) $cal_por_dia[$f] = [];
        $cal_por_dia[$f][] = [
            'id'           => (int)$row['id'],
            'estado'       => $row['estado'],
            'tipo_permiso' => $row['tipo_permiso'],
            'funcionario'  => $row['funcionario'],
            'fecha_sol'    => $row['fecha_solicitud'],
            'fecha_salida' => $row['fecha_salida'],
        ];
    }
    $stmt->close();
}

// helper de clase CSS por estado
function estado_to_chip_class($estado) {
    $e = strtolower($estado);
    if (strpos($e, 'pendiente') !== false)                         return 'chip--pendiente';
    if (strpos($e, 'aprobada por jefe') !== false)                 return 'chip--aprobada-jefe';
    if (strpos($e, 'aprobada por recursos humanos') !== false)     return 'chip--aprobada-rrhh';
    if (strpos($e, 'rechazada por jefe') !== false)                return 'chip--rechazada-jefe';
    if (strpos($e, 'rechazada por recursos humanos') !== false)    return 'chip--rechazada-rrhh';
    return 'chip--pendiente';
}

// Obtener últimas solicitudes (SIN CAMBIOS)
$sql_ultimas = "SELECT s.*, a.Nom AS nombre_funcionario 
                FROM solicitudes s 
                JOIN accounts a ON s.id_usuario = a.ID 
                WHERE s.estado IN ('Aprobada por Jefe Inmediato', 'Rechazada por Jefe Inmediato', 'Aprobada por Recursos Humanos', 'Rechazada por Recursos Humanos') 
                AND a.cod_unidad = ? 
                ORDER BY s.id DESC LIMIT 5";
if ($stmt = $conn->prepare($sql_ultimas)) {
    $stmt->bind_param('s', $cod_unidad_jefe);
    $stmt->execute();
    $result_ultimas = $stmt->get_result();
    $stmt->close();
}

$conn->close();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Panel Principal - Jefe de Departamento</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        * { margin:0; padding:0; box-sizing:border-box; font-family:'Roboto',sans-serif; }
        body { background-color:#f0f4f8; color:#2c3e50; line-height:1.6; }
        .container { display:flex; min-height:100vh; overflow:hidden; }

        /* Sidebar */
        .sidebar { width:260px; background:linear-gradient(135deg,#2c3e50,#34495e); color:white; padding:20px 15px; position:fixed; height:100%; transition:transform .3s ease,width .3s ease; z-index:1000; }
        .sidebar.collapsed { transform:translateX(-260px); width:0; }
        .sidebar h2 { font-size:24px; margin-bottom:30px; text-align:center; color:#68c8d8; }
        .sidebar ul { list-style:none; }
        .sidebar ul li { margin-bottom:15px; }
        .sidebar ul li a { color:#ecf0f1; text-decoration:none; font-size:16px; display:flex; align-items:center; padding:10px 15px; border-radius:5px; transition:background .3s,color .3s; }
        .sidebar ul li a:hover { background:#465c71; color:#68c8d8; }
        .sidebar ul li a i { margin-right:10px; }
        .toggle-btn { display:none; position:fixed; left:10px; top:10px; background:#68c8d8; color:white; border:none; padding:10px; border-radius:5px; cursor:pointer; z-index:1001; }

        /* Main */
        .main-content { margin-left:260px; padding:30px; width:calc(100% - 260px); transition:margin-left .3s ease,width .3s ease; }
        .welcome-message { font-size:28px; color:#2c3e50; margin-bottom:25px; text-align:center; font-weight:500; }
        .main-content h1 { font-size:32px; color:#2c3e50; margin-bottom:30px; text-align:center; border-bottom:3px solid #68c8d8; padding-bottom:10px; font-weight:700; }

        .dashboard-grid { display:grid; grid-template-columns:2fr 1fr; gap:30px; margin-bottom:20px; }
        .top-sections { display:grid; gap:20px; margin-bottom:30px; }

        .request-section, .notifications-section, .calendar-section, .pending-requests, .recent-activity {
            background:white; padding:20px; border-radius:10px; box-shadow:0 4px 12px rgba(0,0,0,.1);
        }
        .request-section h2, .notifications-section h2, .calendar-section h2 { font-size:22px; color:#2c3e50; margin-bottom:15px; }

        .request-buttons { display:flex; justify-content:center; gap:15px; flex-wrap:wrap; }
        .request-buttons button { padding:12px 20px; background:linear-gradient(135deg,#68c8d8,#5dade2); color:white; border:none; border-radius:8px; font-size:16px; cursor:pointer; transition:transform .2s, background .3s; }
        .request-buttons button:hover:not(:disabled) { background:linear-gradient(135deg,#5dade2,#3498db); transform:translateY(-2px); }
        .request-buttons button:disabled { background:#b0bec5; cursor:not-allowed; opacity:.7; }

        .notifications-section ul { list-style:none; }
        .notifications-section li { padding:12px 15px; margin-bottom:10px; border-left:5px solid #f39c12; background:#fffaf0; border-radius:5px; font-size:15px; color:#555; transition:background .3s; }
        .notifications-section li:hover { background:#fef9f3; }

        /* ====== CALENDARIO NUEVO ====== */
        .calendar-header { display:flex; justify-content:space-between; align-items:center; margin-bottom:20px; flex-wrap:wrap; gap:10px; }
        .calendar-header form { display:flex; gap:10px; flex-wrap:wrap; }
        .calendar-header select { padding:8px; border:1px solid #ddd; border-radius:6px; font-size:14px; background:white; transition:border-color .3s; }
        .calendar-header select:focus { border-color:#68c8d8; outline:none; }

        .calendar { display:grid; grid-template-columns:repeat(7,1fr); gap:8px; }
        .calendar .day-header { background:#2c3e50; color:#fff; font-weight:600; text-align:center; padding:10px 6px; border-radius:8px; }

        .day-card { 
            position:relative; display:flex; flex-direction:column; gap:6px; 
            background:#f9fbfd; border:1px solid #ecf0f1; border-radius:10px; 
            padding:10px; text-align:left; min-height:88px; cursor:pointer;
            transition:box-shadow .2s, transform .1s; 
        }
        .day-card:hover { box-shadow:0 6px 16px rgba(0,0,0,.08); transform:translateY(-1px); }
        .day-card:disabled { cursor:default; }

        .date-badge { font-weight:700; color:#2c3e50; font-size:14px; }
        .state-chips { display:flex; flex-wrap:wrap; gap:6px; }
        .chip { display:inline-flex; align-items:center; gap:6px; padding:3px 8px; border-radius:999px; font-size:11px; font-weight:600; border:1px solid transparent; }
        .chip .dot { width:8px; height:8px; border-radius:50%; display:inline-block; }

        .chip--pendiente        { background:#fff6e5; color:#8a5a00; border-color:#ffdf9c; }
        .chip--pendiente .dot   { background:#f5a623; }

        .chip--aprobada-jefe        { background:#e7f1ff; color:#0f4f9a; border-color:#bcd6ff; }
        .chip--aprobada-jefe .dot   { background:#2f80ed; }

        .chip--aprobada-rrhh        { background:#e8f7ef; color:#1f6b3b; border-color:#bfe7cf; }
        .chip--aprobada-rrhh .dot   { background:#27ae60; }

        .chip--rechazada-jefe        { background:#ffecec; color:#8a1f1f; border-color:#ffc4c4; }
        .chip--rechazada-jefe .dot   { background:#e74c3c; }

        .chip--rechazada-rrhh        { background:#ffe8e8; color:#7a1c1c; border-color:#ffbcbc; }
        .chip--rechazada-rrhh .dot   { background:#c0392b; }

        .chip.more { background:#edeff3; color:#34495e; border-color:#d6dbe3; }

        .day-card[data-tt]:hover::after{
            content: attr(data-tt);
            position:absolute; left:8px; right:8px; bottom:8px;
            background:rgba(33, 37, 41, .95); color:#fff; padding:8px 10px; 
            border-radius:8px; font-size:12px; white-space:pre-line;
            box-shadow:0 8px 24px rgba(0,0,0,.3); z-index:20;
        }

        .cards-section { display:grid; grid-template-columns:repeat(auto-fit,minmax(200px,1fr)); gap:20px; }
        .card { background:white; padding:20px; border-radius:10px; box-shadow:0 4px 12px rgba(0,0,0,.1); text-align:center; transition:transform .2s; }
        .card:hover { transform:translateY(-5px); }
        .card h3 { font-size:18px; color:#7f8c8d; margin-bottom:10px; }
        .card p { font-size:32px; font-weight:700; color:#2c3e50; }
        .card.pendientes { border-left:5px solid #f39c12; }
        .card.aprobadas-jefe { border-left:5px solid #3498db; }
        .card.aprobadas-rrhh { border-left:5px solid #27ae60; }
        .card.rechazadas-jefe { border-left:5px solid #e74c3c; }
        .card.rechazadas-rrhh { border-left:5px solid #c0392b; }
        .card.urgentes { border-left:5px solid #e74c3c; }

        .pending-requests h2, .recent-activity h2 { font-size:22px; color:#2c3e50; margin-bottom:15px; }
        .pending-requests table, .recent-activity table { width:100%; border-collapse:collapse; }
        .pending-requests th, .pending-requests td, .recent-activity th, .recent-activity td { padding:12px; text-align:left; border-bottom:1px solid #ecf0f1; }
        .pending-requests th, .recent-activity th { background:#2c3e50; color:white; }
        .pending-requests td, .recent-activity td { color:#7f8c8d; }

        .modal { display:none; position:fixed; top:0; left:0; width:100%; height:100%; background:rgba(0,0,0,.6); backdrop-filter:blur(4px); justify-content:center; align-items:center; z-index:3000; }
        .modal-content { background:#fff; padding:30px; border-radius:15px; width:90%; max-width:620px; box-shadow:0 6px 25px rgba(0,0,0,.3); position:relative; animation:modalScale .3s ease; }
        @keyframes modalScale { from{ transform:scale(.8); opacity:0; } to{ transform:scale(1); opacity:1; } }
        .modal-content h2 { font-size:22px; color:#e74c3c; margin-bottom:20px; text-align:center; font-weight:600; }
        .modal-content h2 i { margin-right:8px; }
        .modal-content .close { position:absolute; top:15px; right:20px; font-size:24px; color:#888; cursor:pointer; transition:color .3s; }
        .modal-content .close:hover { color:#e74c3c; }

        .day-table { width:100%; border-collapse:collapse; }
        .day-table th, .day-table td { padding:10px; border-bottom:1px solid #eee; text-align:left; font-size:14px; }
        .day-table th { background:#f7f9fc; font-weight:700; color:#2c3e50; }
        .badge-sm { padding:2px 8px; border-radius:999px; border:1px solid #e2e8f0; font-size:11px; font-weight:600; display:inline-flex; align-items:center; gap:6px; }
        .badge-sm .dot { width:8px; height:8px; border-radius:50%; display:inline-block; }

        @media (max-width:1024px){ .sidebar{ width:220px; } .main-content{ margin-left:220px; width:calc(100% - 220px); padding:25px; } .dashboard-grid{ grid-template-columns:1fr; } }
        @media (max-width:768px){ .toggle-btn{ display:block; } .sidebar{ transform:translateX(-260px); } .sidebar.open{ transform:translateX(0); } .main-content{ margin-left:0; width:100%; padding:20px; } .dashboard-grid{ grid-template-columns:1fr; } .cards-section{ grid-template-columns:1fr 1fr; } .calendar-header{ flex-direction:column; align-items:flex-start; } }
        @media (max-width:480px){ .cards-section{ grid-template-columns:1fr; } .calendar .day-card{ min-height:92px; } }

        .floating-btn{ position:fixed; bottom:25px; right:25px; background:linear-gradient(135deg,#e74c3c,#c0392b); color:#fff; border:none; padding:15px; border-radius:50%; font-size:20px; cursor:pointer; box-shadow:0 0 15px rgba(231,76,60,.8); transition:background .3s, transform .2s; z-index:2000; animation:pulseGlowRed 1.8s infinite; }
        .floating-btn:hover{ background:linear-gradient(135deg,#c0392b,#96281b); transform:translateY(-3px); }
        @keyframes pulseGlowRed{ 0%{ box-shadow:0 0 10px rgba(231,76,60,.8), 0 0 20px rgba(231,76,60,.5);} 50%{ box-shadow:0 0 20px rgba(231,76,60,1), 0 0 40px rgba(231,76,60,.7);} 100%{ box-shadow:0 0 10px rgba(231,76,60,.8), 0 0 20px rgba(231,76,60,.5);} }
    </style>
</head>
<body>
    <button class="toggle-btn" onclick="toggleSidebar()"><i class="fas fa-bars"></i></button>
    <div class="container">
        <!-- Menú lateral -->
        <div class="sidebar" id="sidebar">
            <h2>Panel Jefe</h2>
            <ul>
                <li><a href="dashboard_jefe.php"><i class="fas fa-tachometer-alt"></i> Panel Principal</a></li>
                <li><a href="panel_jefe.php"><i class="fas fa-clipboard-list"></i> Solicitudes Pendientes-Permiso</a></li>
                <li><a href="panel_jefe_vacaciones.php"><i class="fas fa-umbrella-beach"></i> Solicitudes Pendientes-Vacaciones</a></li>
                <li><a href="solicitudes_aprobadas.php"><i class="fas fa-check"></i> Solicitudes Aprobadas</a></li>
                <li><a href="solicitudes_rechazadas.php"><i class="fas fa-times"></i> Solicitudes Rechazadas</a></li>
                <li><a href="vacaciones_jefe.php"><i class="fas fa-users"></i> Vacaciones</a></li>
                <li><a href="panel_jefe_user.php" class="perfil-link"><i class="fas fa-eye"></i> Perfil</a></li>
                <li>
                    <a href="https://sistemas.proasol.gob.hn/denuncias_proasol/index.html?src=jefe_sidebar" target="_blank" rel="noopener">
                        <i class="fas fa-shield-alt"></i> Denuncia-CPEP
                    </a>
                </li>
                <li><a href="logout.php"><i class="fas fa-sign-out-alt"></i> Cerrar Sesión</a></li>
            </ul>
        </div>

        <!-- Contenido principal -->
        <div class="main-content">
            <div class="welcome-message">Bienvenido, <?= htmlspecialchars($nombre_usuario) ?> <span style="font-size:14px; color:#7f8c8d;"></span></div>
            <h1>Panel Principal - Jefe de Unidad</h1>

            <div class="top-sections">
                <!-- Request Section -->
                <div class="request-section">
                    <h2><i class="fas fa-plus"></i> Nueva Solicitud de Permiso</h2>
                    <div class="request-buttons">
                        <button onclick="window.location.href='pase_salida.php'"><i class="fas fa-plus"></i> Generar Permiso</button>
                        <button onclick="window.location.href='pase_vacaciones.php'"><i class="fas fa-umbrella-beach"></i> Generar Vacaciones</button>
                        <button onclick="openModal()"><i class="fas fa-ellipsis-h"></i> Más Solicitudes</button>
                    </div>
                </div>

                <!-- Notifications -->
                <div class="notifications-section">
                    <h2><i class="fas fa-bell"></i> Notificaciones</h2>
                    <ul>
                        <?php if ($urgentes > 0): ?>
                            <li>⚠️ Hay <?= $urgentes ?> solicitudes pendientes con fechas de salida dentro de los próximos 3 días.</li>
                        <?php endif; ?>
                        <?php if ($pendientes > 0): ?>
                            <li>Tienes <strong><?= $pendientes ?></strong> solicitudes pendientes en <?= htmlspecialchars($nombre_unidad_jefe) ?>.</li>
                        <?php endif; ?>
                        <?php if ($urgentes == 0 && $pendientes == 0): ?>
                            <li>ℹ️ No hay notificaciones nuevas.</li>
                        <?php endif; ?>
                    </ul>
                </div>
            </div>

            <!-- Dashboard Grid -->
            <div class="dashboard-grid">
                <!-- Calendar -->
                <div class="calendar-section">
                    <h2><i class="fas fa-calendar-alt"></i> Calendario de Permisos</h2>
                    <div class="calendar-header">
                        <form method="GET" action="dashboard_jefe.php">
                            <select name="month" onchange="this.form.submit()">
                                <?php
                                $months = [1=>'Enero',2=>'Febrero',3=>'Marzo',4=>'Abril',5=>'Mayo',6=>'Junio',7=>'Julio',8=>'Agosto',9=>'Septiembre',10=>'Octubre',11=>'Noviembre',12=>'Diciembre'];
                                foreach ($months as $num => $name) {
                                    $selected = $selected_month == $num ? 'selected' : '';
                                    echo "<option value='$num' $selected>$name</option>";
                                }
                                ?>
                            </select>
                            <select name="year" onchange="this.form.submit()">
                                <?php
                                for ($year = date('Y') + 1; $year >= date('Y') - 5; $year--) {
                                    $selected = $selected_year == $year ? 'selected' : '';
                                    echo "<option value='$year' $selected>$year</option>";
                                }
                                ?>
                            </select>
                        </form>
                    </div>

                    <!-- GRID DEL CALENDARIO -->
                    <div class="calendar">
                        <div class="day-header">Lun</div><div class="day-header">Mar</div><div class="day-header">Mié</div>
                        <div class="day-header">Jue</div><div class="day-header">Vie</div><div class="day-header">Sáb</div><div class="day-header">Dom</div>
                        <?php
                        // Ajuste para iniciar en Lunes
                        $first_day_of_month = ($first_day_of_month + 6) % 7;
                        for ($i = 0; $i < $first_day_of_month; $i++) echo "<div></div>";

                        for ($day = 1; $day <= $days_in_month; $day++) {
                            $current_date = sprintf('%04d-%02d-%02d', $selected_year, $selected_month, $day);
                            $items = isset($cal_por_dia[$current_date]) ? $cal_por_dia[$current_date] : [];
                            $total = count($items);

                            $tooltip = '';
                            if ($total > 0) {
                                $lines = [];
                                foreach ($items as $it) {
                                    $lines[] = "• {$it['funcionario']} — {$it['tipo_permiso']} ({$it['estado']})";
                                    if (count($lines) >= 8) { $lines[] = "… y ".($total - 8)." más"; break; }
                                }
                                $tooltip = htmlspecialchars(implode("\n", $lines));
                            }

                            echo "<button type='button' class='day-card' data-date='{$current_date}' ".($tooltip ? "data-tt=\"{$tooltip}\"" : "").">";
                                echo "<div class='date-badge'>{$day}</div>";

                                if ($total > 0) {
                                    echo "<div class='state-chips'>";
                                    $shown = 0;
                                    foreach ($items as $it) {
                                        if ($shown >= 3) break;
                                        $class = estado_to_chip_class($it['estado']);
                                        $title = htmlspecialchars("{$it['funcionario']} — {$it['tipo_permiso']} ({$it['estado']})");
                                        echo "<span class='chip {$class}' title='{$title}'>";
                                        echo     "<span class='dot'></span><span>".htmlspecialchars($it['tipo_permiso'])."</span>";
                                        echo "</span>";
                                        $shown++;
                                    }
                                    if ($total > $shown) {
                                        echo "<span class='chip more' title='Haz clic para ver más'><strong>+".($total - $shown)."</strong></span>";
                                    }
                                    echo "</div>";
                                }
                            echo "</button>";
                        }
                        ?>
                    </div>
                </div>

                <!-- Stats -->
                <div class="cards-section">
                    <div class="card pendientes"><h3>Pendientes</h3><p><?= $pendientes ?></p></div>
                    <div class="card aprobadas-jefe"><h3>Aprobadas por Jefe</h3><p><?= $aprobadas_jefe ?></p></div>
                    <div class="card aprobadas-rrhh"><h3>Aprobadas por RRHH</h3><p><?= $aprobadas_rrhh ?></p></div>
                    <div class="card rechazadas-jefe"><h3>Rechazadas por Jefe</h3><p><?= $rechazadas_jefe ?></p></div>
                    <div class="card rechazadas-rrhh"><h3>Rechazadas por RRHH</h3><p><?= $rechazadas_rrhh ?></p></div>
                    <div class="card urgentes"><h3>Urgentes</h3><p><?= $urgentes ?></p></div>
                </div>
            </div>

            <!-- Actividad reciente -->
            <div class="recent-activity">
                <h2><i class="fas fa-history"></i> Actividad Reciente</h2>
                <?php if ($result_ultimas->num_rows > 0): ?>
                    <table>
                        <tr><th>Funcionario</th><th>Fecha Salida</th><th>Estado</th></tr>
                        <?php while ($row = $result_ultimas->fetch_assoc()): ?>
                            <tr>
                                <td><?= htmlspecialchars($row['nombre_funcionario']) ?></td>
                                <td><?= htmlspecialchars($row['fecha_salida']) ?></td>
                                <td><?= htmlspecialchars($row['estado']) ?></td>
                            </tr>
                        <?php endwhile; ?>
                    </table>
                <?php else: ?>
                    <p>No hay actividad reciente.</p>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Modal "Más Solicitudes" -->
    <div id="moreRequestsModal" class="modal">
        <div class="modal-content">
            <span class="close" onclick="closeModal()">×</span>
            <h2><i class="fas fa-plus"></i> Más Solicitudes</h2>
            <form id="moreRequestsForm" action="procesar_solicitud_documento_jefe.php" method="POST">
                <label for="tipo_solicitud">Tipo de Solicitud:</label>
                <select id="tipo_solicitud" name="tipo_solicitud" required>
                    <option value="">Seleccione una opción</option>
                    <option value="Constancia de trabajo">Constancia de trabajo</option>
                    <option value="Constancia de trabajo con deducciones">Constancia de trabajo con deducciones</option>
                    <option value="Constancia para Embajadas">Constancia para Embajadas</option>
                    <option value="Declaración de impuestos">Declaración de impuestos</option>
                    <option value="Voucher de pagos">Voucher de pagos</option>
                    <option value="Otros">Otros</option>
                </select>
                <label for="comentarios">Comentarios:</label>
                <textarea id="comentarios" name="comentarios" placeholder="Escribe un comentario breve sobre lo que estás solicitando"></textarea>
                <div class="modal-buttons">
                    <button type="button" class="cancel" onclick="closeModal()">Cancelar</button>
                    <button type="submit" class="submit">Enviar Solicitud</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Modal de Detalle del Día -->
    <div id="dayDetailModal" class="modal">
        <div class="modal-content">
            <span class="close" onclick="closeDayModal()">×</span>
            <h2><i class="fas fa-calendar-day"></i> Detalle de solicitudes — <span id="dayModalDate"></span></h2>
            <div id="dayModalBody"></div>
        </div>
    </div>

    <!-- Modal Cambiar Contraseña -->
    <div id="passwordModal" class="modal">
        <div class="modal-content">
            <span class="close" onclick="closePasswordModal()">×</span>
            <h2><i class="fas fa-lock"></i> Cambiar Contraseña</h2>
            <form action="cambiar_password.php" method="POST">
                <div class="input-group">
                    <i class="fas fa-key"></i>
                    <input type="password" id="password" name="password" required minlength="6" placeholder="Nueva contraseña">
                </div>
                <div class="input-group">
                    <i class="fas fa-check-circle"></i>
                    <input type="password" id="confirm_password" name="confirm_password" required placeholder="Confirmar contraseña">
                </div>
                <div id="password-alert">⚠️ Las contraseñas no coinciden.</div>
                <div class="modal-buttons">
                    <button type="button" class="cancel" onclick="closePasswordModal()">Cancelar</button>
                    <button type="submit" class="submit">Guardar</button>
                </div>
            </form>
        </div>
    </div>

    <!-- SweetAlert2 -->
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

    <script>
        /* ====== Sidebar ====== */
        function toggleSidebar() {
            const sidebar = document.getElementById('sidebar');
            sidebar.classList.toggle('open');
            const mainContent = document.querySelector('.main-content');
            if (sidebar.classList.contains('open')) {
                mainContent.style.marginLeft = '260px';
                mainContent.style.width = 'calc(100% - 260px)';
            } else {
                mainContent.style.marginLeft = '0';
                mainContent.style.width = '100%';
            }
        }

        /* ====== Modales existentes ====== */
        function openModal() { document.getElementById('moreRequestsModal').style.display = 'flex'; }
        function closeModal() { document.getElementById('moreRequestsModal').style.display = 'none'; document.getElementById('moreRequestsForm').reset(); }

        function openPasswordModal() {
            document.getElementById('passwordModal').style.display = 'flex';
        }
        function closePasswordModal() { document.getElementById('passwordModal').style.display = 'none'; }

        document.querySelector("form[action='cambiar_password.php']").addEventListener("submit", function(e) {
            const pass = document.getElementById("password").value;
            const confirm = document.getElementById("confirm_password").value;
            const alertBox = document.getElementById("password-alert");
            if (pass !== confirm) { e.preventDefault(); alertBox.style.display = "block"; }
            else { alertBox.style.display = "none"; }
        });

        /* ====== Datos del calendario (inyectados por PHP) ====== */
        window.CAL_DATA = <?php
            $safe = [];
            foreach ($cal_por_dia as $fecha => $arr) {
                $tmp = [];
                foreach ($arr as $it) {
                    $tmp[] = [
                        'id'           => (int)$it['id'],
                        'estado'       => $it['estado'],
                        'tipo_permiso' => $it['tipo_permiso'],
                        'funcionario'  => $it['funcionario'],
                        'fecha_sol'    => $it['fecha_sol'],
                        'fecha_salida' => $it['fecha_salida'],
                    ];
                }
                $safe[$fecha] = $tmp;
            }
            echo json_encode($safe, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
        ?>;

        /* ====== Modal de detalle por día ====== */
        const dayModal = document.getElementById('dayDetailModal');
        const dayModalDate = document.getElementById('dayModalDate');
        const dayModalBody = document.getElementById('dayModalBody');

        function closeDayModal(){ dayModal.style.display = 'none'; }

        function dotColor(estado){
            const e = (estado || '').toLowerCase();
            if (e.includes('pendiente')) return '#f5a623';
            if (e.includes('aprobada por jefe')) return '#2f80ed';
            if (e.includes('aprobada por recursos humanos')) return '#27ae60';
            if (e.includes('rechazada por jefe')) return '#e74c3c';
            if (e.includes('rechazada por recursos humanos')) return '#c0392b';
            return '#6c757d';
        }

        document.querySelectorAll('.day-card').forEach(btn => {
            btn.addEventListener('click', () => {
                const date = btn.getAttribute('data-date');
                const items = (window.CAL_DATA && window.CAL_DATA[date]) ? window.CAL_DATA[date] : [];
                dayModalDate.textContent = new Date(date).toLocaleDateString('es-HN', { weekday:'long', year:'numeric', month:'long', day:'numeric' });

                if (!items.length) {
                    dayModalBody.innerHTML = '<p style="text-align:center; color:#6b7280;">No hay solicitudes en esta fecha.</p>';
                } else {
                    let rows = items.map(it => `
                        <tr>
                            <td><strong>${escapeHtml(it.funcionario)}</strong></td>
                            <td>${escapeHtml(it.tipo_permiso || '')}</td>
                            <td>
                                <span class="badge-sm">
                                    <span class="dot" style="background:${dotColor(it.estado)}"></span>
                                    ${escapeHtml(it.estado || '')}
                                </span>
                            </td>
                            <td>${escapeHtml(it.fecha_sol || '')}</td>
                            <td>${escapeHtml(it.fecha_salida || '')}</td>
                        </tr>
                    `).join('');
                    dayModalBody.innerHTML = `
                        <div style="overflow:auto;">
                            <table class="day-table">
                                <thead>
                                    <tr>
                                        <th>Funcionario</th>
                                        <th>Tipo</th>
                                        <th>Estado</th>
                                        <th>Fecha Solicitud</th>
                                        <th>Fecha Salida</th>
                                    </tr>
                                </thead>
                                <tbody>${rows}</tbody>
                            </table>
                        </div>
                    `;
                }
                dayModal.style.display = 'flex';
            });
        });

        window.addEventListener('click', (e) => {
            if (e.target === dayModal) closeDayModal();
        });

        function escapeHtml(str){
            if (str == null) return '';
            return String(str)
                .replace(/&/g,'&amp;')
                .replace(/</g,'&lt;')
                .replace(/>/g,'&gt;')
                .replace(/"/g,'&quot;')
                .replace(/'/g,'&#039;');
        }

        /* ====== Persistencia por usuario ====== */
        const UID = <?= json_encode($user_id) ?>;
        const k = (suffix) => `u:${UID}:${suffix}`;
        const markProfile  = () => localStorage.setItem(k('didProfile'), '1');
        const markPassword = () => localStorage.setItem(k('didPassword'), '1');
        const hasProfile   = () => localStorage.getItem(k('didProfile')) === '1';
        const hasPassword  = () => localStorage.getItem(k('didPassword')) === '1';
        const isComplete   = () => hasProfile() && hasPassword();

        document.querySelectorAll("a[href='panel_jefe_user.php']").forEach(a => {
            a.addEventListener('click', () => { markProfile(); });
        });

        /* ====== CONTROL DE INACTIVIDAD DE SESIÓN ======
           Cambia INACTIVITY_LIMIT_MS si quieres más/menos minutos
        */
        const INACTIVITY_LIMIT_MS = 5 * 60 * 1000; // 15 minutos
        let inactivityTimer = null;
        let sessionExpired = false;

        function startInactivityTimer() {
            clearTimeout(inactivityTimer);
            if (sessionExpired) return;

            inactivityTimer = setTimeout(() => {
                sessionExpired = true;
                Swal.fire({
                    title: 'Sesión suspendida',
                    text: 'Tu sesión ha sido suspendida por inactividad. Por seguridad debes iniciar sesión nuevamente.',
                    icon: 'warning',
                    confirmButtonText: 'Aceptar'
                }).then(() => {
                    window.location.href = 'logout.php?reason=timeout';
                });
            }, INACTIVITY_LIMIT_MS);
        }

        function resetInactivityTimer() {
            if (sessionExpired) return;
            startInactivityTimer();
        }

        // Eventos que cuentan como actividad del usuario
        ['load', 'mousemove', 'mousedown', 'click', 'scroll', 'keypress', 'touchstart', 'touchmove'].forEach(evt => {
            window.addEventListener(evt, resetInactivityTimer, true);
        });

        // Iniciar el contador la primera vez
        startInactivityTimer();
    </script>

    <!-- Botón flotante fijo -->
    <button class="floating-btn" onclick="openPasswordModal()">
        <i class="fas fa-key"></i>
    </button>
</body>
</html>
