<?php
session_start();
require 'conexion.php';

// Verificar que el usuario esté autenticado
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

$user_id = $_SESSION['user_id'];

// Obtener el nombre completo del usuario desde la tabla accounts
$sql_usuario = "SELECT Nom, Nom2, Ape, Ape2 FROM accounts WHERE ID = ?";
if ($stmt = $conn->prepare($sql_usuario)) {
    $stmt->bind_param('i', $user_id);
    $stmt->execute();
    $stmt->bind_result($nom, $nom2, $ape, $ape2);
    $stmt->fetch();
    $nombre_completo = trim(implode(' ', array_filter([$nom, $nom2, $ape, $ape2])));
    $stmt->close();
} else {
    $nombre_completo = "Error al obtener el nombre";
}

// Verificar que se haya pasado un ID de solicitud válido
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    $_SESSION['error'] = 'ID de solicitud no válido.';
    header("Location: historial_solicitudes.php");
    exit;
}

$solicitud_id = (int)$_GET['id'];

// Obtener los datos de la solicitud, incluyendo la firma del funcionario
$sql_solicitud = "SELECT tipo_permiso, fecha_solicitud, fecha_salida, descripcion_motivo, estado, 
                        unidad, puesto,  horario_permiso, 
                        firma_funcionario, firma_jefe, firma_rrhh,
                        fecha_aprobacion_jefe, fecha_aprobacion_rrhh,nombre_funcionario_firma,nombre_jefe_firma,nombre_rrhh_firma
                 FROM solicitudes 
                 WHERE id = ? AND id_usuario = ?";

if ($stmt = $conn->prepare($sql_solicitud)) {
    $stmt->bind_param('ii', $solicitud_id, $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $solicitud = $result->fetch_assoc();
    $stmt->close();
    
    if (!$solicitud) {
        $_SESSION['error'] = 'Solicitud no encontrada o no tienes permiso para verla.';
        header("Location: historial_solicitudes.php");
        exit;
    }
} else {
    $_SESSION['error'] = 'Error al obtener la solicitud.';
    header("Location: historial_solicitudes.php");
    exit;
}

// Formatear fechas para el formato datetime-local
$fecha_solicitud = date('Y-m-d\TH:i', strtotime($solicitud['fecha_solicitud']));
$fecha_salida = date('Y-m-d\TH:i', strtotime($solicitud['fecha_salida']));

// Usar horario_permiso directamente desde la tabla
$horario_permiso = $solicitud['horario_permiso'] ?? "N/A";

$conn->close();
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Detalle de Solicitud - Pase de Salida</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            background-color: #f5f5f5;
            margin: 0;
            padding: 0;
        }

        .container {
            max-width: 800px;
            margin: 20px auto;
            background-color: white;
            padding: 20px;
            border: 1px solid #ccc;
            box-shadow: 0 0 10px rgba(0, 0, 0, 0.1);
            position: relative;
        }

        .ribbon {
            position: absolute;
            top: 0;
            left: 0;
            width: 10px;
            height: 100%;
            background-color: #68c8d8;
        }

        header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }

        header h1 {
            font-size: 24px;
            margin: 0;
            color: #333;
        }

        .logo-left img {
            width: 100px;
            height: 100px;
        }

        .logo-right img {
            width: 150px;
            height: 100px;
        }

        .form-group {
            margin-bottom: 15px;
        }

        .form-group label {
            display: block;
            font-weight: bold;
            margin-bottom: 5px;
        }

        .form-group input,
        .form-group textarea,
        .form-group select {
            width: 100%;
            padding: 8px;
            border: 1px solid #ccc;
            border-radius: 4px;
            box-sizing: border-box;
        }

        .form-group textarea {
            resize: vertical;
        }

        .form-group.inline {
            display: flex;
            gap: 20px;
        }

        .form-group.inline > div {
            flex: 1;
        }

        .form-group.inline .motivo {
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .form-group.inline .motivo label {
            margin-bottom: 0;
        }

        .form-group.inline .motivo input[type="radio"] {
            margin: 0 5px 0 0;
        }

        .signature-line {
            border-bottom: 1px solid #000;
            height: 20px;
            margin-top: 10px;
        }

        .signatures {
            margin-top: 30px;
        }

        .jefe {
            text-align: center;
            margin-top: 30px;
            margin-left: auto;
            margin-right: auto;
        }

        .jefe p {
            margin: 5px 0;
            font-weight: bold;
        }

        .observaciones h3 {
            margin-bottom: 10px;
            font-size: 16px;
        }

        .observaciones ul {
            list-style-type: disc;
            padding-left: 20px;
        }

        .observaciones li {
            margin-bottom: 10px;
        }

        footer {
            text-align: center;
            margin-top: 30px;
            font-size: 12px;
            color: #666;
        }

        footer p {
            margin: 5px 0;
        }

        .print-button {
            display: block;
            margin: 20px auto;
            padding: 10px 20px;
            background-color: #68c8d8;
            color: white;
            border: none;
            border-radius: 5px;
            font-size: 16px;
            cursor: pointer;
            transition: background 0.3s;
        }

        .print-button:hover {
            background-color: #56b0c0;
        }

        .signature-pad {
            border: none;
            height: 150px;
            margin-top: 10px;
            width: 100%;
        }

        .signature-image {
            height: 150px;
            margin-top: 10px;
            width: 100%;
            border: none;
            object-fit: contain;
        }

        /* Estilos para impresión */
        @media print {
            body {
                margin: 0;
                padding: 0;
                background: none;
                -webkit-print-color-adjust: exact !important;
                print-color-adjust: exact !important;
            }

            .signature-pad {
                height: 80px !important;
                width: 100% !important;
                border: none !important;
            }

            .signature-image {
                height: 80px !important;
                width: 100% !important;
                border: none !important;
            }

            #descripcion {
                height: 40px !important;
                font-size: 10px !important;
                padding: 4px !important;
            }

            .jefe {
                margin-top: 10px !important;
                width: 60% !important;
            }

            .jefe canvas {
                height: 60px !important;
                margin-top: 5px !important;
            }

            @page {
                size: A4;
                margin: 5mm;
            }

            .container {
                width: 200mm;
                min-height: 287mm;
                margin: 0;
                padding: 10mm;
                background-color: white !important;
                border: 1px solid #ccc !important;
                box-shadow: none;
                position: relative;
                display: flex;
                flex-direction: column;
                justify-content: space-between;
            }

            .ribbon {
                position: absolute;
                top: 0;
                left: 0;
                width: 10px;
                height: 100%;
                background-color: #68c8d8 !important;
            }

            header {
                display: flex;
                justify-content: space-between;
                align-items: center;
                margin-bottom: 25px;
            }

            header h1 {
                font-size: 28px;
                margin: 0;
                color: #333;
            }

            .logo-left img {
                width: 120px;
                height: 120px;
            }

            .logo-right img {
                width: 180px;
                height: 120px;
            }

            .form-group {
                margin-bottom: 20px;
                page-break-inside: avoid;
            }

            .form-group label {
                display: block;
                font-weight: bold;
                margin-bottom: 6px;
                font-size: 14px;
            }

            .form-group input,
            .form-group textarea,
            .form-group select {
                width: 100%;
                padding: 10px;
                border: 1px solid #ccc !important;
                border-radius: 4px;
                box-sizing: border-box;
                background: none;
                color: #000 !important;
                font-size: 14px;
            }

            .form-group textarea {
                resize: none;
                height: 80px;
            }

            .form-group.inline {
                display: flex;
                gap: 25px;
            }

            .form-group.inline > div {
                flex: 1;
            }

            .form-group.inline .motivo {
                display: flex;
                align-items: center;
                gap: 10px;
            }

            .form-group.inline .motivo label {
                margin-bottom: 0;
            }

            .form-group.inline .motivo input[type="radio"] {
                margin: 0 5px 0 0;
            }

            .signature-line {
                border-bottom: 1px solid #000 !important;
                height: 25px;
                margin-top: 12px;
            }

            .signatures {
                margin-top: 40px;
                page-break-inside: avoid;
            }

            .jefe {
                text-align: center;
                margin-top: 40px;
                page-break-inside: avoid;
            }

            .jefe p {
                margin: 6px 0;
                font-weight: bold;
                color: #000 !important;
                font-size: 14px;
            }

            .observaciones {
                flex-grow: 1;
            }

            .observaciones h3 {
                margin-bottom: 12px;
                font-size: 12px;
                color: #000 !important;
            }

            .observaciones ul {
                list-style-type: disc;
                padding-left: 20px;
            }

            .observaciones li {
                margin-bottom: 8px;
                color: #666 !important;
                font-size: 10px;
            }

            footer {
                text-align: center;
                margin-top: 30px;
                font-size: 10px;
                color: #666 !important;
                page-break-inside: avoid;
            }

            footer p {
                margin: 6px 0;
            }

            .print-button {
                display: none !important;
            }
        }

        .signatures {
    display: flex;
    gap: 40px;
    justify-content: space-around;
    margin-top: 20px;
}

.signature-block {
    display: flex;
    flex-direction: column;
    align-items: center;
    border: 1px solid #ddd;
    padding: 15px 20px;
    border-radius: 8px;
    background-color: #f9f9f9;
    min-width: 180px;
    box-shadow: 0 2px 5px rgba(0,0,0,0.1);
}

.signature-block label {
    font-size: 1rem;
    margin-bottom: 5px;
}

.signature-block .name {
    font-weight: bold;
    margin-bottom: 5px;
}

.signature-block .date {
    font-size: 0.9rem;
    color: #555;
}


    </style>
</head>
<body>
    <div class="container">
        <div class="ribbon"></div>
        <header>
            <div class="logo-left">
                <img src="logo-proasol.png" alt="Logo PROASOL">
            </div>
            <h1>Pase de Salida</h1>
            <div class="logo-right">
                <img src="gobierno-logo.png" alt="Logo HONDUR">
            </div>
        </header>
        <div>
            <div class="form-group">
                <label for="nombre-funcionario">Nombre del Funcionario</label>
                <input type="text" id="nombre-funcionario" name="nombre_funcionario" value="<?= htmlspecialchars($nombre_completo) ?>" readonly>
            </div>

            <div class="form-group inline">
                <div>
                    <label for="unidad">Unidad</label>
                    <input type="text" id="unidad" name="unidad" value="<?= htmlspecialchars($solicitud['unidad'] ?? 'N/A') ?>" readonly>
                </div>
                <div>
                    <label for="puesto">Puesto</label>
                    <input type="text" id="puesto" name="puesto" value="<?= htmlspecialchars($solicitud['puesto'] ?? 'N/A') ?>" readonly>
                </div>
            </div>

            <div class="form-group">
                <label for="nombre-jefe">Nombre del Jefe Inmediato</label>
                <input type="text" id="nombre-jefe" name="nombre_jefe_inmediato" value="<?= htmlspecialchars($solicitud['nombre_jefe_inmediato'] ?? 'N/A') ?>" readonly>
            </div>

            <div class="form-group inline">
                <div>
                    <label for="fecha-solicitud">Fecha de Solicitud</label>
                    <input type="datetime-local" id="fecha-solicitud" name="fecha_solicitud" value="<?= $fecha_solicitud ?>" readonly>
                </div>
                <div class="motivo">
                    <label for="tipo_permiso">Tipo de Permiso:</label>
                    <input type="radio" name="tipo_permiso" value="Personal" <?= $solicitud['tipo_permiso'] === 'Personal' ? 'checked' : '' ?> disabled> Personal
                    <input type="radio" name="tipo_permiso" value="Oficial" <?= $solicitud['tipo_permiso'] === 'Oficial' ? 'checked' : '' ?> disabled> Oficial
                    <input type="radio" name="tipo_permiso" value="Oficial" <?= $solicitud['tipo_permiso'] === 'Vacaciones' ? 'checked' : '' ?> disabled> Vacaciones
                </div>
            </div>

            <div class="form-group">
                <label for="descripcion">Breve Descripción del Motivo del Permiso</label>
                <textarea id="descripcion" name="descripcion_motivo" rows="3" readonly><?= htmlspecialchars($solicitud['descripcion_motivo']) ?></textarea>
            </div>

            <div class="form-group inline">
                <div>
                    <label for="fecha-salida">Fecha de Salida</label>
                    <input type="datetime-local" id="fecha-salida" name="fecha_salida" value="<?= $fecha_salida ?>" readonly>
                </div>
                <div>
                    <label for="horario">Horario del Permiso Solicitado</label>
                    <input type="text" id="horario" name="horario_permiso" value="<?= htmlspecialchars($horario_permiso) ?>" readonly>
                </div>
            </div>
<div class="signatures">
    <!-- Firma del Solicitante -->
    <div class="signature-block">
        <label><strong>Solicitante</strong></label><br>
        <span class="name">
            <?= htmlspecialchars($solicitud['nombre_funcionario_firma'] ?? '---') ?>
            (<?= htmlspecialchars($solicitud['firma_funcionario'] ?? '') ?>)
        </span><br>
        <span class="date">Fecha: <?= !empty($solicitud['fecha_solicitud']) ? date('d-m-Y H:i', strtotime($solicitud['fecha_solicitud'])) : '---' ?></span>
    </div>

    <!-- Firma del Jefe Inmediato -->
    <div class="signature-block">
        <label><strong>Jefe Inmediato</strong></label><br>
        <span class="name">
            <?= htmlspecialchars($solicitud['nombre_jefe_firma'] ?? '---') ?>
            (<?= htmlspecialchars($solicitud['firma_jefe'] ?? '') ?>)
        </span><br>
        <span class="date">Fecha: <?= !empty($solicitud['fecha_aprobacion_jefe']) ? date('d-m-Y H:i', strtotime($solicitud['fecha_aprobacion_jefe'])) : '---' ?></span>
    </div>

    <!-- Firma de RRHH -->
    <div class="signature-block">
        <label><strong>RRHH</strong></label><br>
        <span class="name">
            <?= htmlspecialchars($solicitud['nombre_rrhh_firma'] ?? '---') ?>
            (<?= htmlspecialchars($solicitud['firma_rrhh'] ?? '') ?>)
        </span><br>
        <span class="date">Fecha: <?= !empty($solicitud['fecha_aprobacion_rrhh']) ? date('d-m-Y H:i', strtotime($solicitud['fecha_aprobacion_rrhh'])) : '---' ?></span>
    </div>
</div>





            <div class="form-group observaciones">
                <h3>Observaciones:</h3>
                <ul>
                    <li>Ausentarse o abandonar su puesto en horarios laborales sin la debida autorización se considerará una falta leve.</li>
                    <li>Se podrán otorgar hasta tres autorizaciones justificadas por mes y serán válidas por un máximo de dos horas si el asunto es personal.</li>
                    <li>Para asuntos oficiales, las autorizaciones se otorgarán por el tiempo que sea necesario y se deberán presentar los respaldos correspondientes.</li>
                    <li>Independientemente el motivo del permiso deberá marcar en el reloj biométrico.</li>
                    <li>Si el permiso se encuentra dentro del horario laboral deberá marcar la salida y el regreso a su puesto de trabajo.</li>
                </ul>
            </div>

            <footer>
                <p>CENTRO CÍVICO GUBERNAMENTAL, TORRE 2, PISO 1, BOULEVARD JUAN PABLO II, TEGUCIGALPA, M.D.C., HONDURAS, C.A.</p>
                <p>(+504) 2242-7981</p>
            </footer>
        </div>
        <button class="print-button" onclick="window.print()">Imprimir Pase</button>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/signature_pad@4.0.0/dist/signature_pad.umd.min.js"></script>
    <script>
        // Inicializar los lienzos de firmas (solo para Jefe y RRHH, ya que la firma del Funcionario se muestra como imagen)
        const signaturePadJefe = new SignaturePad(document.getElementById('firma-jefe'));
        const signaturePadGina = new SignaturePad(document.getElementById('firma-gina'));
    </script>
</body>
</html>