<?php
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

require 'conexion.php';

// Verificar que el usuario esté autenticado y tenga el rol correcto (cod_rol = 8)
if (!isset($_SESSION['user_id']) || !isset($_SESSION['role']) || $_SESSION['role'] != 8) {
    $_SESSION['error'] = 'Acceso no autorizado. Por favor, inicia sesión con una cuenta de Jefe de Departamento válida.';
    header("Location: login.php");
    exit;
}

$user_id = $_SESSION['user_id'];

// Obtener el cod_unidad del jefe
$sql_jefe_unidad = "SELECT cod_unidad FROM accounts WHERE ID = ?";
$cod_unidad_jefe = null;
if ($stmt = $conn->prepare($sql_jefe_unidad)) {
    $stmt->bind_param('i', $user_id);
    $stmt->execute();
    $stmt->bind_result($cod_unidad_jefe);
    $stmt->fetch();
    $stmt->close();
}

if (is_null($cod_unidad_jefe)) {
    $_SESSION['error'] = 'No tienes una unidad asignada. Contacta al administrador.';
    header("Location: solicitudes_aprobadas.php");
    exit;
}

// Verificar que se haya pasado un ID de solicitud válido
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    $_SESSION['error'] = 'ID de solicitud no válido.';
    header("Location: solicitudes_aprobadas.php");
    exit;
}

$solicitud_id = (int)$_GET['id'];

// Obtener los datos de la solicitud, incluyendo la firma del funcionario
$sql_solicitud = "SELECT tipo_permiso, fecha_solicitud, fecha_salida, descripcion_motivo, estado, 
                        unidad, puesto, nombre_jefe_inmediato, horario_permiso, firma_funcionario, 
                        firma_jefe, firma_rrhh, fecha_aprobacion_jefe, fecha_aprobacion_rrhh,
                        nombre_funcionario_firma, nombre_jefe_firma, nombre_rrhh_firma
                 FROM solicitudes 
                 WHERE id = ? AND id_usuario IN (SELECT ID FROM accounts WHERE cod_unidad = ?)";

if ($stmt = $conn->prepare($sql_solicitud)) {
    $stmt->bind_param('is', $solicitud_id, $cod_unidad_jefe);
    $stmt->execute();
    $result = $stmt->get_result();
    $solicitud = $result->fetch_assoc();
    $stmt->close();
    
    if (!$solicitud) {
        $_SESSION['error'] = 'Solicitud no encontrada o no tienes permiso para verla.';
        header("Location: solicitudes_aprobadas.php");
        exit;
    }
} else {
    $_SESSION['error'] = 'Error al obtener la solicitud.';
    header("Location: solicitudes_aprobadas.php");
    exit;
}

$datos_vacaciones = null;

if ($solicitud['tipo_permiso'] === 'Vacaciones') {
    $sql_vacaciones = "SELECT fecha_inicio, fecha_fin, dias 
                       FROM vacaciones 
                       WHERE id_solicitud = ?";

    if ($stmt = $conn->prepare($sql_vacaciones)) {
        $stmt->bind_param('i', $solicitud_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $datos_vacaciones = $result->fetch_assoc(); // Solo trae la primera fila
        $stmt->close();
    }
}

// Formatear fechas para el formato datetime-local
$fecha_solicitud = date('Y-m-d\TH:i', strtotime($solicitud['fecha_solicitud']));
$fecha_salida = date('Y-m-d\TH:i', strtotime($solicitud['fecha_salida']));

// Usar horario_permiso directamente desde la tabla
$horario_permiso = $solicitud['horario_permiso'] ?? "N/A";

// Obtener el nombre completo del usuario desde la tabla accounts
$sql_usuario = "SELECT Nom, Nom2, Ape, Ape2 FROM accounts WHERE ID = (
    SELECT id_usuario FROM solicitudes WHERE id = ?
)";
if ($stmt = $conn->prepare($sql_usuario)) {
    $stmt->bind_param('i', $solicitud_id);
    $stmt->execute();
    $stmt->bind_result($nom, $nom2, $ape, $ape2);
    $stmt->fetch();
    $nombre_completo = trim(implode(' ', array_filter([$nom, $nom2, $ape, $ape2])));
    $stmt->close();
} else {
    $nombre_completo = "Error al obtener el nombre";
}

$conn->close();
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Detalle de Solicitud - Pase de Salida</title>
    <style>
    body {
    font-family: Arial, sans-serif;
    background-color: #f5f5f5;
    margin: 0;
    padding: 0;
}

.container {
    max-width: 800px;
    margin: 20px auto;
    background-color: white;
    padding: 20px;
    border: 1px solid #ccc;
    box-shadow: 0 0 10px rgba(0, 0, 0, 0.1);
    position: relative;
}

.ribbon {
    position: absolute;
    top: 0;
    left: 0;
    width: 10px;
    height: 100%;
    background-color: #68c8d8;
}

header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
}

header h1 {
    font-size: 24px;
    margin: 0;
    color: #333;
    text-align: center;
    flex: 1;
}

.logo-left img {
    width: 100px;
    height: 100px;
    object-fit: contain;
}

.logo-right img {
    width: 150px;
    height: 100px;
    object-fit: contain;
}

.form-group {
    margin-bottom: 15px;
}

.form-group label {
    display: block;
    font-weight: bold;
    margin-bottom: 5px;
}

.form-group input,
.form-group textarea,
.form-group select {
    width: 100%;
    padding: 8px;
    border: 1px solid #ccc;
    border-radius: 4px;
    box-sizing: border-box;
}

.form-group textarea {
    resize: vertical;
}

.form-group.inline {
    display: flex;
    gap: 20px;
}

.form-group.inline > div {
    flex: 1;
}

/* ------------------------------ */
/* NUEVO: RADIO BUTTONS HORIZONTALES */
/* ------------------------------ */

.permiso-horizontal {
    display: flex;
    align-items: center;
    gap: 20px;
    flex-wrap: wrap;
}

.permiso-horizontal .titulo {
    font-weight: bold;
    margin-right: 10px;
}

.permiso-horizontal .radio-op {
    display: flex;
    align-items: center;
    gap: 5px;
    font-size: 15px;
}

.permiso-horizontal input[type="radio"] {
    transform: scale(1.1);
}

/* ------------------------------ */

.signature-line {
    border-bottom: 1px solid #000;
    height: 20px;
    margin-top: 10px;
}

.signatures {
    display: flex;
    gap: 40px;
    justify-content: space-around;
    margin-top: 20px;
}

.signature-block {
    display: flex;
    flex-direction: column;
    align-items: center;
    border: 1px solid #ddd;
    padding: 15px 20px;
    border-radius: 8px;
    background-color: #f9f9f9;
    min-width: 180px;
    box-shadow: 0 2px 5px rgba(0,0,0,0.1);
}

.signature-block label {
    font-size: 1rem;
    margin-bottom: 5px;
}

.signature-block .name {
    font-weight: bold;
    margin-bottom: 5px;
    text-align: center;
}

.signature-block .date {
    font-size: 0.9rem;
    color: #555;
}

.observaciones h3 {
    margin-bottom: 10px;
    font-size: 16px;
}

.observaciones ul {
    list-style-type: disc;
    padding-left: 20px;
}

.observaciones li {
    margin-bottom: 10px;
}

footer {
    text-align: center;
    margin-top: 30px;
    font-size: 12px;
    color: #666;
}

footer p {
    margin: 5px 0;
}

/* ------------------------------ */
/* BOTONES */
/* ------------------------------ */

.buttons-container {
    display: flex;
    justify-content: center;
    gap: 10px;
    margin-top: 20px;
    flex-wrap: wrap;
}

.print-button,
.back-button {
    padding: 10px 20px;
    border: none;
    border-radius: 5px;
    font-size: 16px;
    cursor: pointer;
    transition: background 0.3s, transform 0.1s;
}

.print-button {
    background-color: #68c8d8;
    color: white;
}

.print-button:hover {
    background-color: #56b0c0;
    transform: translateY(-1px);
}

.back-button {
    background-color: #e74c3c;
    color: #fff;
}

.back-button:hover {
    background-color: #c0392b;
    transform: translateY(-1px);
}

/* ------------------------------ */
/* RESPONSIVE */
/* ------------------------------ */

@media (max-width: 767px) {

    .container {
        margin: 10px;
        padding: 15px;
        max-width: 100%;
        box-shadow: none;
        border: none;
    }

    header {
        flex-direction: column;
        align-items: center;
        gap: 15px;
        text-align: center;
    }

    .logo-left img,
    .logo-right img {
        width: 80px;
        height: auto;
    }

    header h1 {
        font-size: 20px;
    }

    .form-group.inline {
        flex-direction: column;
        gap: 10px;
    }

    .form-group.inline > div {
        flex: none;
        width: 100%;
    }

    .signatures {
        flex-direction: column;
        gap: 20px;
        align-items: center;
    }

    .signature-block {
        min-width: auto;
        width: 100%;
    }

    .print-button,
    .back-button {
        width: 100%;
        font-size: 18px;
    }

    .buttons-container {
        flex-direction: column;
    }

    .permiso-horizontal .titulo {
        width: 100%;
        text-align: center;
    }
}

/* ========================================================= */
/*                     MODO IMPRESIÓN                       */
/* ========================================================= */

@media print {

    html, body {
        width: 210mm;
        height: 297mm;
        margin: 0;
        padding: 0;
        background: none;
        font-size: 11px;
        -webkit-print-color-adjust: exact !important;
        print-color-adjust: exact !important;
    }

    @page {
        size: A4 portrait;
        margin: 8mm;
    }

    /* ------------------------------ */
    /* 🔥 NUEVO: HEADER SIEMPRE HORIZONTAL AL IMPRIMIR */
    /* ------------------------------ */

    header {
        display: flex !important;
        flex-direction: row !important;
        justify-content: space-between !important;
        align-items: center !important;
        width: 100% !important;
        gap: 10px !important;
        margin-bottom: 15px !important;
        page-break-inside: avoid !important;
    }

    header h1 {
        flex: 1 !important;
        text-align: center !important;
        font-size: 18px !important;
        margin: 0 !important;
        padding: 0 !important;
    }

    .logo-left img {
        width: 80px !important;
        height: 80px !important;
        object-fit: contain !important;
        display: block !important;
    }

    .logo-right img {
        width: 110px !important;
        height: 80px !important;
        object-fit: contain !important;
        display: block !important;
    }

    /* ------------------------------ */

    .container {
        width: 100%;
        max-width: 190mm;
        margin: 0;
        padding: 8mm;
        background-color: white !important;
        border: 1px solid #ccc !important;
        box-shadow: none;
        display: block;
        page-break-inside: avoid;
    }

    .ribbon {
        position: absolute;
        top: 0;
        left: 0;
        width: 8px;
        height: 100%;
        background-color: #68c8d8 !important;
    }

    .form-group {
        margin-bottom: 8px;
        page-break-inside: avoid;
    }

    .form-group label {
        margin-bottom: 3px;
        font-size: 11px;
    }

    .form-group input,
    .form-group textarea,
    .form-group select {
        padding: 4px;
        font-size: 11px;
        border: 1px solid #ccc !important;
        background: none;
        color: #000 !important;
    }

    #descripcion {
        height: 40px !important;
        font-size: 10px !important;
        padding: 4px !important;
    }

    .signature-pad,
    .signature-image {
        height: 80px !important;
        width: 100% !important;
        border: none !important;
    }

    .signatures {
        margin-top: 10px;
        gap: 15px;
    }

    .signature-block {
        min-width: 0;
        padding: 8px 10px;
        box-shadow: none;
        background-color: #f9f9f9 !important;
    }

    footer {
        margin-top: 10px;
        font-size: 9px;
        color: #666 !important;
    }

    footer p {
        margin: 3px 0;
    }

    /* Ocultar botones */
    .buttons-container,
    .print-button,
    .back-button {
        display: none !important;
    }
}

    </style>
</head>
<body>
    <div class="container">
        <div class="ribbon"></div>
        <header>
            <div class="logo-left">
                <img src="logo-proasol.png" alt="Logo PROASOL">
            </div>
            <h1>Detalle de Solicitud</h1>
            <div class="logo-right">
                <img src="gobierno-logo.png" alt="Logo HONDUR">
            </div>
        </header>
        <div>
            <div class="form-group">
                <label for="nombre-funcionario">Nombre del Funcionario</label>
                <input type="text" id="nombre-funcionario" name="nombre_funcionario" value="<?= htmlspecialchars($nombre_completo) ?>" readonly>
            </div>

            <div class="form-group inline">
                <div>
                    <label for="unidad">Unidad</label>
                    <input type="text" id="unidad" name="unidad" value="<?= htmlspecialchars($solicitud['unidad'] ?? 'N/A') ?>" readonly>
                </div>
                <div>
                    <label for="puesto">Puesto</label>
                    <input type="text" id="puesto" name="puesto" value="<?= htmlspecialchars($solicitud['puesto'] ?? 'N/A') ?>" readonly>
                </div>
            </div>

            <div class="form-group">
                <label for="nombre-jefe">Nombre del Jefe Inmediato</label>
                <input type="text" id="nombre-jefe" name="nombre_jefe_inmediato" value="<?= htmlspecialchars($solicitud['nombre_jefe_inmediato'] ?? 'N/A') ?>" readonly>
            </div>

            <div class="form-group inline">
                <div>
                    <label for="fecha-solicitud">Fecha de Solicitud</label>
                    <input type="datetime-local" id="fecha-solicitud" name="fecha_solicitud" value="<?= $fecha_solicitud ?>" readonly>
                </div>
                <div class="motivo permiso-horizontal">
                    <label class="titulo">Tipo de Permiso:</label>

                    <label class="radio-op">
                        <input type="radio" name="tipo_permiso" value="Personal"
                        <?= $solicitud['tipo_permiso'] === 'Personal' ? 'checked' : '' ?> disabled>
                        Personal
                    </label>

                    <label class="radio-op">
                        <input type="radio" name="tipo_permiso" value="Oficial"
                        <?= $solicitud['tipo_permiso'] === 'Oficial' ? 'checked' : '' ?> disabled>
                        Oficial
                    </label>

                    <label class="radio-op">
                        <input type="radio" name="tipo_permiso" value="Vacaciones"
                        <?= $solicitud['tipo_permiso'] === 'Vacaciones' ? 'checked' : '' ?> disabled>
                        Vacaciones
                    </label>
                </div>
            </div>

            <div class="form-group">
                <label for="descripcion">Breve Descripción del Motivo del Permiso</label>
                <textarea id="descripcion" name="descripcion_motivo" rows="3" readonly><?= htmlspecialchars($solicitud['descripcion_motivo']) ?></textarea>
            </div>

            <div class="form-group inline">
                <div>
                    <label for="fecha-salida">Fecha de Salida</label>
                    <input type="datetime-local" id="fecha-salida" name="fecha_salida" value="<?= $fecha_salida ?>" readonly>
                </div>
                <div>
                    <label for="horario">Horario / Días del Permiso Solicitado</label>
                    <input type="text" id="horario" name="horario_permiso" value="<?= htmlspecialchars($horario_permiso) ?>" readonly>
                </div>
            </div>
            
            <div id="campos-vacaciones" style="<?= $solicitud['tipo_permiso'] === 'Vacaciones' ? '' : 'display: none;' ?>">
                <div class="form-group inline">
                    <div>
                        <label for="fecha_inicio">Fecha Inicio de Vacaciones</label>
                        <input type="date" id="fecha_inicio" value="<?= htmlspecialchars($datos_vacaciones['fecha_inicio'] ?? '') ?>" readonly>
                    </div>
                    <div>
                        <label for="fecha_fin">Fecha Final de Vacaciones</label>
                        <input type="date" id="fecha_fin" value="<?= htmlspecialchars($datos_vacaciones['fecha_fin'] ?? '') ?>" readonly>
                    </div>
                </div>
                <div class="form-group">
                    <label for="dias">Días Solicitados</label>
                    <input type="number" id="dias" value="<?= htmlspecialchars($datos_vacaciones['dias'] ?? '') ?>" readonly>
                </div>
            </div>

            <div class="signatures">
                <!-- Firma del Solicitante -->
                <div class="signature-block">
                    <label><strong>Solicitante</strong></label><br>
                    <span class="name">
                        <?= htmlspecialchars($solicitud['nombre_funcionario_firma'] ?? '---') ?>
                        (<?= htmlspecialchars($solicitud['firma_funcionario'] ?? '') ?>)
                    </span><br>
                    <span class="date">Fecha: <?= !empty($solicitud['fecha_solicitud']) ? date('d-m-Y H:i', strtotime($solicitud['fecha_solicitud'])) : '---' ?></span>
                </div>

                <!-- Firma del Jefe Inmediato -->
                <div class="signature-block">
                    <label><strong>Jefe Inmediato</strong></label><br>
                    <span class="name">
                        <?= htmlspecialchars($solicitud['nombre_jefe_firma'] ?? '---') ?>
                        (<?= htmlspecialchars($solicitud['firma_jefe'] ?? '') ?>)
                    </span><br>
                    <span class="date">Fecha: <?= !empty($solicitud['fecha_aprobacion_jefe']) ? date('d-m-Y H:i', strtotime($solicitud['fecha_aprobacion_jefe'])) : '---' ?></span>
                </div>

                <!-- Firma de RRHH -->
                <div class="signature-block">
                    <label><strong>RRHH</strong></label><br>
                    <span class="name">
                        <?= htmlspecialchars($solicitud['nombre_rrhh_firma'] ?? '---') ?>
                        (<?= htmlspecialchars($solicitud['firma_rrhh'] ?? '') ?>)
                    </span><br>
                    <span class="date">Fecha: <?= !empty($solicitud['fecha_aprobacion_rrhh']) ? date('d-m-Y H:i', strtotime($solicitud['fecha_aprobacion_rrhh'])) : '---' ?></span>
                </div>
            </div>

            <div class="form-group observaciones">
                <h3>Observaciones:</h3>
                <ul>
                    <li>Ausentarse o abandonar su puesto en horarios laborales sin la debida autorización se considerará una falta leve.</li>
                    <li>Se podrán otorgar hasta tres autorizaciones justificadas por mes y serán válidas por un máximo de dos horas si el asunto es personal.</li>
                    <li>Para asuntos oficiales, las autorizaciones se otorgarán por el tiempo que sea necesario y se deberán presentar los respaldos correspondientes.</li>
                    <li>Independientemente del motivo del permiso deberá marcar en el reloj biométrico.</li>
                    <li>Si el permiso se encuentra dentro del horario laboral deberá marcar la salida y el regreso a su puesto de trabajo.</li>
                </ul>
            </div>

            <footer>
                <p>CENTRO CÍVICO GUBERNAMENTAL, TORRE 2, PISO 1, BOULEVARD JUAN PABLO II, TEGUCIGALPA, M.D.C., HONDURAS, C.A.</p>
                <p>(+504) 2242-7981</p>
            </footer>
        </div>

        <div class="buttons-container">
            <button type="button"
                    class="back-button"
                    onclick="window.location.href='solicitudes_aprobadas.php'">
                Regresar
            </button>

            <button type="button"
                    class="print-button"
                    onclick="window.print()">
                Imprimir Pase
            </button>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/signature_pad@4.0.0/dist/signature_pad.umd.min.js"></script>
    <script>
        // Aquí podrías inicializar signature_pad si tuvieras canvas para firmas
        // const signaturePadJefe = new SignaturePad(document.getElementById('firma-jefe'));
        // const signaturePadGina = new SignaturePad(document.getElementById('firma-gina'));
    </script>
    
    <script>
        document.addEventListener("DOMContentLoaded", function () {
            const radios = document.querySelectorAll('input[name="tipo_permiso"]');
            const camposVacaciones = document.getElementById('campos-vacaciones');

            radios.forEach(radio => {
                radio.addEventListener('change', function () {
                    if (this.value === "Vacaciones") {
                        camposVacaciones.style.display = "block";
                    } else {
                        camposVacaciones.style.display = "none";
                    }
                });
            });
        });
    </script>

</body>
</html>
