<?php
require 'conexion.php';

$unidad       = $_POST['unidad']        ?? '';
$empleado     = $_POST['empleado']      ?? '';
$fecha_inicio = $_POST['fecha_inicio']  ?? '';
$fecha_fin    = $_POST['fecha_fin']     ?? '';

$condiciones = [];
$parametros  = [];
$tipos       = '';

$filtraPorRango = ($fecha_inicio !== '' && $fecha_fin !== '');

/* Filtros por unidad y empleado: se aplican sobre la vista unificada (alias t) */
if ($unidad !== '') {
    $condiciones[] = "t.cod_unidad = ?";
    $parametros[]  = $unidad;
    $tipos        .= 's';
}
if ($empleado !== '') {
    $condiciones[] = "t.id_usuario = ?";
    $parametros[]  = $empleado;
    $tipos        .= 'i';
}

/*
 Subconsulta base que unifica fechas de "ejecución":
 - Vacaciones: rango MIN(fecha_inicio)..MAX(fecha_fin), suma de días (dias_vac)
 - Personal/Oficial: fecha_salida (inicio=fin=fecha_salida), mantiene horario_permiso
 - Otros: cae a fecha_solicitud (inicio=fin)
*/
$sql_base = "
    SELECT
        s.id                           AS id_solicitud,
        s.id_usuario,
        a.cod_unidad,
        CONCAT(a.Nom, ' ', COALESCE(a.Nom2, ''), ' ', a.Ape, ' ', COALESCE(a.Ape2, '')) AS nombre_empleado,

        DATE(s.fecha_solicitud)        AS fecha_solicitud,
        s.tipo_permiso,
        DATE(s.fecha_salida)           AS fecha_salida,

        CASE WHEN s.tipo_permiso IN ('Personal','Oficial') THEN s.horario_permiso ELSE NULL END AS horario_permiso,

        v.fecha_inicio_vac             AS fecha_inicio_vac,
        v.fecha_fin_vac                AS fecha_fin_vac,
        v.dias_vac                     AS dias_vac,

        s.estado,

        /* Inicio efectivo */
        CASE
            WHEN s.tipo_permiso = 'Vacaciones' AND v.fecha_inicio_vac IS NOT NULL THEN v.fecha_inicio_vac
            WHEN s.tipo_permiso IN ('Personal','Oficial') AND s.fecha_salida IS NOT NULL THEN DATE(s.fecha_salida)
            ELSE DATE(s.fecha_solicitud)
        END AS fecha_evento_inicio,

        /* Fin efectivo */
        CASE
            WHEN s.tipo_permiso = 'Vacaciones' AND v.fecha_fin_vac IS NOT NULL THEN v.fecha_fin_vac
            WHEN s.tipo_permiso IN ('Personal','Oficial') AND s.fecha_salida IS NOT NULL THEN DATE(s.fecha_salida)
            ELSE DATE(s.fecha_solicitud)
        END AS fecha_evento_fin
    FROM solicitudes s
    INNER JOIN accounts a ON s.id_usuario = a.ID
    LEFT JOIN (
        SELECT 
            id_solicitud,
            MIN(fecha_inicio) AS fecha_inicio_vac,
            MAX(fecha_fin)    AS fecha_fin_vac,
            SUM(dias)         AS dias_vac
        FROM vacaciones
        GROUP BY id_solicitud
    ) v ON v.id_solicitud = s.id
";

/*
 Selección final:
 - Formatea fechas
 - Incluye dias_en_rango (si hay filtro, intersección con el rango; si no, días del evento)
 - Expone dias_vac y mapea luego un campo 'dias' en PHP para compatibilidad con tu tabla
*/
$sql = "SELECT 
            t.nombre_empleado,
            t.id_solicitud,
            t.id_usuario,
            t.cod_unidad,
            t.tipo_permiso,
            t.estado,
            DATE_FORMAT(t.fecha_solicitud,  '%Y-%m-%d') AS fecha_solicitud,
            DATE_FORMAT(t.fecha_salida,     '%Y-%m-%d') AS fecha_salida,
            DATE_FORMAT(t.fecha_inicio_vac, '%Y-%m-%d') AS fecha_inicio,
            DATE_FORMAT(t.fecha_fin_vac,    '%Y-%m-%d') AS fecha_fin,

            DATE_FORMAT(t.fecha_evento_inicio, '%Y-%m-%d') AS fecha_evento_inicio,
            DATE_FORMAT(t.fecha_evento_fin,    '%Y-%m-%d') AS fecha_evento_fin,

            t.horario_permiso,
            t.dias_vac,
            " . (
                $filtraPorRango
                ? "GREATEST(0, DATEDIFF(LEAST(t.fecha_evento_fin, ?), GREATEST(t.fecha_evento_inicio, ?)) + 1)"
                : "GREATEST(0, DATEDIFF(t.fecha_evento_fin, t.fecha_evento_inicio) + 1)"
            ) . " AS dias_en_rango
        FROM ( $sql_base ) t
";

/* Filtro por solapamiento de rango cuando RRHH envía fechas:
   Entra si: inicio_evento <= fin_filtro  AND  fin_evento >= inicio_filtro
   OJO con el orden de parámetros: primero van los ? del SELECT (si hay rango),
   luego los de WHERE (unidad, empleado, y por último los 2 del solapamiento).
*/
if ($filtraPorRango) {
    // Estos 2 parámetros son para el cálculo de dias_en_rango en el SELECT (LEAST/GREATEST)
    $paramsSelect = [$fecha_fin, $fecha_inicio];
    $typesSelect  = 'ss';

    // Añade condición de solapamiento al final del WHERE
    $condiciones[] = "t.fecha_evento_inicio <= ? AND t.fecha_evento_fin >= ?";
    $paramsWhereRango = [$fecha_fin, $fecha_inicio];
    $typesWhereRango  = 'ss';
} else {
    $paramsSelect    = [];
    $typesSelect     = '';
    $paramsWhereRango = [];
    $typesWhereRango  = '';
}

$where = count($condiciones) ? " WHERE " . implode(" AND ", $condiciones) : "";
$sql .= $where . " ORDER BY t.fecha_evento_inicio DESC";

$stmt = $conn->prepare($sql);

/* Construcción del arreglo final de parámetros respetando el orden de aparición de ?:
   1) ? del SELECT (si hay rango)
   2) ? del WHERE por unidad/empleado (en el orden en que se añadieron)
   3) ? del WHERE por solapamiento (si hay rango)
*/
$allParams = [];
$allTypes  = '';

if ($filtraPorRango) {
    $allParams = array_merge($allParams, $paramsSelect);
    $allTypes .= $typesSelect;
}
if (!empty($parametros)) {
    $allParams = array_merge($allParams, $parametros);
    $allTypes .= $tipos;
}
if ($filtraPorRango) {
    $allParams = array_merge($allParams, $paramsWhereRango);
    $allTypes .= $typesWhereRango;
}

if (!empty($allParams)) {
    $stmt->bind_param($allTypes, ...$allParams);
}

$stmt->execute();
$result = $stmt->get_result();

$data = [];
while ($row = $result->fetch_assoc()) {
    // Normaliza strings
    $row['nombre_empleado']     = $row['nombre_empleado']     ?? '';
    $row['tipo_permiso']        = $row['tipo_permiso']        ?? '';
    $row['estado']              = $row['estado']              ?? '';

    $row['fecha_solicitud']     = $row['fecha_solicitud']     ?? '';
    $row['fecha_salida']        = $row['fecha_salida']        ?? '';
    $row['fecha_inicio']        = $row['fecha_inicio']        ?? '';
    $row['fecha_fin']           = $row['fecha_fin']           ?? '';

    $row['fecha_evento_inicio'] = $row['fecha_evento_inicio'] ?? '';
    $row['fecha_evento_fin']    = $row['fecha_evento_fin']    ?? '';

    $row['horario_permiso']     = $row['horario_permiso']     ?? '';

    // Normaliza numéricos
    if (isset($row['dias_en_rango']) && is_numeric($row['dias_en_rango'])) {
        $row['dias_en_rango'] = 0 + $row['dias_en_rango'];
    } else {
        $row['dias_en_rango'] = null;
    }
    if (isset($row['dias_vac']) && is_numeric($row['dias_vac'])) {
        $row['dias_vac'] = 0 + $row['dias_vac'];
    } else {
        $row['dias_vac'] = null;
    }

    /* 🔑 Compatibilidad para la tabla: siempre expón 'dias'
       - Si hay filtro de fechas y el evento cae dentro, usa dias_en_rango
       - Si no, usa el total de días de vacaciones de la solicitud (dias_vac)
       - Último recurso: si es evento de 1 día (Personal/Oficial), usar 1
    */
    if ($row['dias_en_rango'] !== null) {
        $row['dias'] = $row['dias_en_rango'];
    } elseif ($row['dias_vac'] !== null) {
        $row['dias'] = $row['dias_vac'];
    } else {
        $row['dias'] = ($row['fecha_evento_inicio'] && $row['fecha_evento_fin'] && $row['fecha_evento_inicio'] === $row['fecha_evento_fin']) ? 1 : 0;
    }

    $data[] = $row;
}

echo json_encode($data);

$stmt->close();
$conn->close();
