<?php
session_start();
require 'conexion.php';

// Verificar que el usuario esté autenticado
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

$user_id = $_SESSION['user_id'];

// Obtener el nombre completo del usuario desde la tabla accounts
$sql_usuario = "SELECT Nom, Nom2, Ape, Ape2 FROM accounts WHERE ID = ?";
if ($stmt = $conn->prepare($sql_usuario)) {
    $stmt->bind_param('i', $user_id);
    $stmt->execute();
    $stmt->bind_result($nom, $nom2, $ape, $ape2);
    $stmt->fetch();

    // Combinar los campos para formar el nombre completo
    $nombre_completo = trim(implode(' ', array_filter([$nom, $nom2, $ape, $ape2])));
    
    $stmt->close();
} else {
    $nombre_completo = "Error al obtener el nombre";
}

// Determinar la página actual
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$records_per_page = 10;
$offset = ($page - 1) * $records_per_page;

/*
 * ============================
 *  TOTAL DE REGISTROS (COUNT)
 * ============================
 * Contamos sólo las solicitudes que cumplan:
 * - tipo_permiso IN ('Personal','Vacaciones','Oficial')
 *   O
 * - tipo_solicitud IN (tus nuevos tipos)
 */
$sql_total = "
    SELECT COUNT(*) 
    FROM solicitudes 
    WHERE id_usuario = ?
      AND (
          tipo_permiso IN ('Personal','Vacaciones','Oficial')
          OR tipo_solicitud IN (
              'Voucher de pagos',
              'Constancia de trabajo con deducciones',
              'Constancia de trabajo',
              'Constancia para Embajadas',
              'Declaracion de impuestos'
          )
      )
";
if ($stmt = $conn->prepare($sql_total)) {
    $stmt->bind_param('i', $user_id);
    $stmt->execute();
    $stmt->bind_result($total_records);
    $stmt->fetch();
    $stmt->close();
} else {
    $total_records = 0;
}

$total_pages = ceil($total_records / $records_per_page);

/*
 * ======================================
 *  OBTENER SOLICITUDES PARA LA PÁGINA
 * ======================================
 * Usamos un alias en la columna tipo_permiso para
 * mostrar tanto permisos (Personal/Vacaciones/Oficial)
 * como los nuevos tipos en tipo_solicitud.
 */
$sql_solicitudes = "
    SELECT 
        id,
        IFNULL(NULLIF(tipo_permiso, ''), tipo_solicitud) AS tipo_permiso,
        fecha_solicitud, 
        descripcion_motivo, 
        estado 
    FROM solicitudes 
    WHERE id_usuario = ? 
      AND (
          tipo_permiso IN ('Personal','Vacaciones','Oficial')
          OR tipo_solicitud IN (
              'Voucher de pagos',
              'Constancia de trabajo con deducciones',
              'Constancia de trabajo',
              'Constancia para Embajadas',
              'Declaracion de impuestos'
          )
      )
    ORDER BY fecha_solicitud DESC
    LIMIT ?, ?
";

if ($stmt = $conn->prepare($sql_solicitudes)) {
    $stmt->bind_param('iii', $user_id, $offset, $records_per_page);
    $stmt->execute();
    $result = $stmt->get_result();
    $solicitudes = $result->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
} else {
    $solicitudes = [];
}

$conn->close();

?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Historial de Solicitudes</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Roboto', sans-serif;
        }

        body {
            background-color: #f0f4f8;
            color: #2c3e50;
            line-height: 1.6;
        }

        .container {
            display: flex;
            min-height: 100vh;
        }

        /* Sidebar */
        .sidebar {
            width: 260px;
            background: linear-gradient(135deg, #2c3e50, #34495e);
            color: white;
            padding: 20px 15px;
            position: fixed;
            height: 100%;
            transition: transform 0.3s ease;
            z-index: 1000;
        }

        .sidebar.collapsed {
            transform: translateX(-260px);
            width: 0;
        }

        .sidebar h2 {
            font-size: 24px;
            margin-bottom: 30px;
            text-align: center;
            color: #68c8d8;
        }

        .sidebar ul {
            list-style: none;
        }

        .sidebar ul li {
            margin-bottom: 15px;
        }

        .sidebar ul li a {
            color: #ecf0f1;
            text-decoration: none;
            font-size: 16px;
            display: flex;
            align-items: center;
            padding: 10px 15px;
            border-radius: 5px;
            transition: background 0.3s, color 0.3s;
        }

        .sidebar ul li a:hover {
            background: #465c71;
            color: #68c8d8;
        }

        .sidebar ul li a i {
            margin-right: 10px;
        }

        .toggle-btn {
            display: none;
            position: fixed;
            left: 10px;
            top: 10px;
            background: #68c8d8;
            color: white;
            border: none;
            padding: 10px;
            border-radius: 5px;
            cursor: pointer;
            z-index: 1001;
            transition: background 0.3s;
        }

        .toggle-btn:hover {
            background: #5dade2;
        }

        /* Main Content */
        .main-content {
            flex-grow: 1;
            padding: 30px;
            margin-left: 260px;
            transition: margin-left 0.3s ease;
            width: calc(100% - 260px);
        }

        .welcome-message {
            font-size: 28px;
            color: #2c3e50;
            margin-bottom: 25px;
            text-align: center;
            font-weight: 500;
        }

        .main-content h1 {
            font-size: 32px;
            color: #2c3e50;
            margin-bottom: 30px;
            text-align: center;
            border-bottom: 3px solid #68c8d8;
            padding-bottom: 10px;
            font-weight: 700;
        }

        /* Sections Grid */
        .dashboard-grid {
            display: grid;
            grid-template-columns: 1fr;
            gap: 30px;
        }

        /* Table Section */
        .table-section {
            background: white;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            overflow-x: auto; /* Permite desplazamiento horizontal en pantallas pequeñas */
        }

        .table-section h2 {
            font-size: 22px;
            color: #2c3e50;
            margin-bottom: 15px;
        }

        table {
            width: 100%;
            min-width: 600px; /* Mínimo ancho para evitar colapso */
            border-collapse: collapse;
            margin-bottom: 20px;
            table-layout: fixed;
        }

        th, td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #ecf0f1;
            word-wrap: break-word; /* Permite que el texto se divida */
        }

        th {
            background: #68c8d8;
            color: white;
            font-weight: 500;
            position: sticky;
            top: 0;
            z-index: 1;
        }

        tr {
            transition: background 0.3s;
        }

        tr:hover {
            background: #f9fbfd;
        }

        .estado-pendiente {
            color: #f39c12;
            font-weight: 500;
        }

        .estado-aprobado {
            color: #27ae60;
            font-weight: 500;
        }

        .estado-rechazado {
            color: #e74c3c;
            font-weight: 500;
        }

        .action-link {
            display: inline-block;
            padding: 8px 12px;
            background: linear-gradient(135deg, #68c8d8, #5dade2);
            color: white;
            text-decoration: none;
            border-radius: 5px;
            font-weight: 500;
            transition: background 0.3s, transform 0.2s;
            white-space: nowrap; /* Evita que se divida en varias líneas */
        }

        .action-link:hover {
            background: linear-gradient(135deg, #5dade2, #3498db);
            transform: translateY(-2px);
        }

        .pagination {
            display: flex;
            justify-content: center;
            gap: 10px;
            margin-top: 20px;
            flex-wrap: wrap; /* Permite que los enlaces se ajusten en móviles */
        }

        .pagination a {
            padding: 8px 12px;
            background: linear-gradient(135deg, #68c8d8, #5dade2);
            color: white;
            text-decoration: none;
            border-radius: 5px;
            transition: background 0.3s;
            font-size: 14px;
        }

        .pagination a:hover {
            background: linear-gradient(135deg, #5dade2, #3498db);
        }

        .pagination a.disabled {
            background: #b0bec5;
            pointer-events: none;
            opacity: 0.7;
        }

        /* Report Button */
        .report-button {
            display: block;
            width: 250px;
            margin: 20px auto;
            padding: 12px;
            background: linear-gradient(135deg, #68c8d8, #5dade2);
            color: white;
            text-align: center;
            text-decoration: none;
            border-radius: 8px;
            font-size: 16px;
            transition: background 0.3s, transform 0.2s;
        }

        .report-button:hover {
            background: linear-gradient(135deg, #5dade2, #3498db);
            transform: translateY(-2px);
        }

        /* Responsive Design */
        @media (max-width: 1024px) {
            .sidebar {
                width: 220px;
            }
            .main-content {
                margin-left: 220px;
                width: calc(100% - 220px);
                padding: 25px;
            }
        }

        @media (max-width: 768px) {
            .toggle-btn {
                display: block;
            }
            .sidebar {
                transform: translateX(-260px);
                width: 260px;
            }
            .sidebar.open {
                transform: translateX(0);
            }
            .main-content {
                margin-left: 0;
                width: 100%;
                padding: 20px;
            }
            .table-section {
                padding: 15px;
            }
            table {
                font-size: 14px;
            }
            th, td {
                padding: 8px;
            }
            .action-link {
                padding: 6px 10px;
                font-size: 12px;
            }
            .pagination a {
                padding: 6px 10px;
                font-size: 12px;
            }
        }

        @media (max-width: 480px) {
            .sidebar {
                padding: 10px;
            }
            .sidebar h2 {
                font-size: 18px;
                margin-bottom: 15px;
            }
            .sidebar ul li a {
                font-size: 12px;
                padding: 6px 10px;
            }
            .main-content {
                padding: 10px;
            }
            .welcome-message {
                font-size: 18px;
            }
            .main-content h1 {
                font-size: 24px;
                margin-bottom: 20px;
            }
            .table-section h2 {
                font-size: 18px;
            }
            th, td {
                padding: 6px;
                font-size: 10px;
            }
            .action-link {
                padding: 4px 8px;
                font-size: 10px;
            }
            .pagination a {
                padding: 4px 8px;
                font-size: 10px;
            }
            .report-button {
                width: 200px;
                font-size: 14px;
                padding: 10px;
            }
        }
    </style>
</head>
<body>
    <button class="toggle-btn" onclick="toggleSidebar()"><i class="fas fa-bars"></i></button>
    <div class="container">
        <!-- Menú lateral -->
        <div class="sidebar" id="sidebar">
            <h2>Panel Solicitante</h2>
            <ul>
                <li><a href="dashboard_usuario.php"><i class="fas fa-tachometer-alt"></i> Inicio</a></li>
                <li><a href="historial_solicitudes.php"><i class="fas fa-list"></i> Mis Solicitudes</a></li>
                <li><a href="panel_empleado.php"><i class="fas fa-user"></i> Perfil</a></li>
                <li><a href="vacaciones.php"><i class="fas fa-umbrella-beach"></i> Vacaciones</a></li>

                <!-- NUEVO: acceso directo al portal de denuncias -->
                <li>
                    <a href="https://sistemas.proasol.gob.hn/denuncias_proasol/index.html?src=myhr_sidebar"
                    target="_blank" rel="noopener">
                    <i class="fas fa-shield-alt"></i> Denuncia-CPEP
                    </a>
                </li>

                <li><a href="logout.php"><i class="fas fa-sign-out-alt"></i> Cerrar Sesión</a></li>
            </ul>
        </div>

        <!-- Contenido principal -->
        <div class="main-content">
            <div class="welcome-message">Bienvenido, <?= htmlspecialchars($nombre_completo) ?> <span style="font-size: 14px; color: #7f8c8d;"></span></div>
            <h1>Historial de Solicitudes</h1>

            <!-- Dashboard Grid -->
            <div class="dashboard-grid">
                <!-- Table Section -->
                <div class="table-section">
                    <h2><i class="fas fa-list"></i> Tus Solicitudes</h2>
                    <table>
                        <thead>
                            <tr>
                                <th>Tipo de Solicitud</th>
                                <th>Fecha de Solicitud</th>
                                <th>Motivo</th>
                                <th>Estado</th>
                                <th>Acción</th>
                            </tr>
                        </thead>
                        <tbody id="solicitudes-table">
                            <?php if (empty($solicitudes)): ?>
                                <tr>
                                    <td colspan="5" style="text-align: center;">No hay solicitudes registradas.</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($solicitudes as $solicitud): ?>
                                    <tr>
                                        <td><?= htmlspecialchars($solicitud['tipo_permiso']) ?></td>
                                        <td><?= htmlspecialchars($solicitud['fecha_solicitud']) ?></td>
                                        <td><?= htmlspecialchars($solicitud['descripcion_motivo']) ?></td>
                                        <td class="estado-<?= strtolower(str_replace(' ', '-', $solicitud['estado'])) ?>">
                                            <?= htmlspecialchars($solicitud['estado']) ?>
                                        </td>
                                        <td>
                                            <a href="detalle_solicitud.php?id=<?= htmlspecialchars($solicitud['id'] ?? '') ?>" class="action-link">Ver Detalles</a>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                    <div class="pagination">
                        <a href="#" class="prev-page <?= $page <= 1 ? 'disabled' : '' ?>" data-page="<?= $page - 1 ?>">Anterior</a>
                        <a href="#" class="next-page <?= $page >= $total_pages ? 'disabled' : '' ?>" data-page="<?= $page + 1 ?>">Siguiente</a>
                    </div>
                </div>
            </div>

           
        </div>
    </div>

    <script>
        function toggleSidebar() {
            const sidebar = document.getElementById('sidebar');
            sidebar.classList.toggle('open');
            const mainContent = document.querySelector('.main-content');
            if (sidebar.classList.contains('open')) {
                mainContent.style.marginLeft = '260px';
                mainContent.style.width = 'calc(100% - 260px)';
            } else {
                mainContent.style.marginLeft = '0';
                mainContent.style.width = '100%';
            }
        }

        document.addEventListener('DOMContentLoaded', function() {
            const prevPageLink = document.querySelector('.prev-page');
            const nextPageLink = document.querySelector('.next-page');

            function loadPage(page) {
                fetch(`get_solicitudes.php?page=${page}`)
                    .then(response => response.json())
                    .then(data => {
                        const tableBody = document.getElementById('solicitudes-table');
                        tableBody.innerHTML = '';

                        if (data.solicitudes.length === 0) {
                            tableBody.innerHTML = '<tr><td colspan="5" style="text-align: center;">No hay solicitudes registradas.</td></tr>';
                        } else {
                            data.solicitudes.forEach(solicitud => {
                                const row = document.createElement('tr');
                                row.innerHTML = `
                                    <td>${solicitud.tipo_permiso}</td>
                                    <td>${solicitud.fecha_solicitud}</td>
                                    <td>${solicitud.descripcion_motivo}</td>
                                    <td class="estado-${solicitud.estado.toLowerCase().replace(' ', '-')}">${solicitud.estado}</td>
                                    <td><a href="detalle_solicitud.php?id=${solicitud.id}" class="action-link">Ver Detalles</a></td>
                                `;
                                tableBody.appendChild(row);
                            });
                        }

                        prevPageLink.classList.toggle('disabled', page <= 1);
                        nextPageLink.classList.toggle('disabled', page >= data.total_pages);
                        prevPageLink.dataset.page = page - 1;
                        nextPageLink.dataset.page = page + 1;

                        // Actualizar URL sin recargar la página
                        history.pushState(null, '', `historial_solicitudes.php?page=${page}`);
                    })
                    .catch(error => console.error('Error al cargar las solicitudes:', error));
            }

            prevPageLink.addEventListener('click', function(e) {
                e.preventDefault();
                const page = parseInt(this.dataset.page);
                if (page >= 1) {
                    loadPage(page);
                }
            });

            nextPageLink.addEventListener('click', function(e) {
                e.preventDefault();
                const page = parseInt(this.dataset.page);
                if (page <= <?= $total_pages ?>) {
                    loadPage(page);
                }
            });
        });
    </script>
</body>
</html>
