<?php
session_start();
require 'conexion.php';

// ✅ Acceso permitido a roles 8 (jefe) y 12
if (!isset($_SESSION['user_id']) || !isset($_SESSION['role']) || ($_SESSION['role'] != 8 && $_SESSION['role'] != 12)) {
    $_SESSION['error'] = 'Acceso no autorizado. Por favor, inicia sesión con una cuenta válida.';
    header("Location: login.php");
    exit;
}

$userId = (int)$_SESSION['user_id'];

// ====== Rango de fechas para vacaciones (ajústalo o hazlo dinámico por GET/POST) ======
$fechaInicio = '2025-07-01';
$fechaFin    = '2026-02-28';

// Obtener cod_unidad del jefe (para fallback por unidad)
$unidadJefe = null;
$stmtUnidad = $conn->prepare("SELECT cod_unidad FROM accounts WHERE ID = ?");
$stmtUnidad->bind_param("i", $userId);
$stmtUnidad->execute();
$stmtUnidad->bind_result($unidadJefe);
$stmtUnidad->fetch();
$stmtUnidad->close();

// =========================
//   Jerarquía (sub-árbol)
//   usando accounts.superior_id
// =========================
function getSubtreeUserIds(mysqli $conn, int $viewerId): array {
    $ids = [$viewerId];
    $q = [$viewerId];
    $stmt = $conn->prepare("SELECT ID FROM accounts WHERE superior_id = ?");
    while ($q) {
        $next = [];
        foreach ($q as $p) {
            $stmt->bind_param('i', $p);
            $stmt->execute();
            $r = $stmt->get_result();
            while ($row = $r->fetch_assoc()) {
                $id = (int)$row['ID'];
                if (!in_array($id, $ids, true)) { $ids[] = $id; $next[] = $id; }
            }
        }
        $q = $next;
    }
    $stmt->close();
    return $ids;
}

$ids = getSubtreeUserIds($conn, $userId);
$ids = array_values(array_unique($ids));
if (count($ids) === 0) { $ids = [$userId]; }
$phIds = implode(',', array_fill(0, count($ids), '?'));

// =========================
//   CONSULTA (UNION de 3 bloques)
//   👉 AHORA TRAE TODAS LAS PENDIENTES (sin MAX por usuario)
//   ✅ Incluye fecha_solicitud y nombre completo como nombre_funcionario
// =========================
$sql = "
  /* Bloque 1: Jerarquía por superior_id (subárbol) */
  (
    SELECT 
      s.id AS id_solicitud, s.fecha_solicitud,
      v.id_vacacion, v.id_usuario, v.fecha_inicio, v.fecha_fin, v.dias,
      CONCAT_WS(' ', a.Nom, a.Nom2, a.Ape, a.Ape2) AS nombre_funcionario
    FROM vacaciones v
    JOIN solicitudes s ON v.id_solicitud = s.id
    JOIN accounts a    ON v.id_usuario   = a.ID
    WHERE s.tipo_permiso = 'Vacaciones'
      AND s.estado = 'Pendiente'
      AND v.fecha_inicio >= ?
      AND v.fecha_fin    <= ?
      AND s.id_usuario IN ($phIds)
      AND a.ID <> ?
  )
  UNION
  /* Bloque 2: Fallback por unidad (roles 7 y 8) */
  (
    SELECT 
      s.id AS id_solicitud, s.fecha_solicitud,
      v.id_vacacion, v.id_usuario, v.fecha_inicio, v.fecha_fin, v.dias,
      CONCAT_WS(' ', a.Nom, a.Nom2, a.Ape, a.Ape2) AS nombre_funcionario
    FROM vacaciones v
    JOIN solicitudes s ON v.id_solicitud = s.id
    JOIN accounts a    ON v.id_usuario   = a.ID
    WHERE s.tipo_permiso = 'Vacaciones'
      AND s.estado = 'Pendiente'
      AND v.fecha_inicio >= ?
      AND v.fecha_fin    <= ?
      AND a.cod_unidad = ?
      AND a.cod_rol IN (7, 8)
      AND a.ID <> ?
  )
  UNION
  /* Bloque 3: Fallback legacy por cod_jefe (reportes directos) */
  (
    SELECT 
      s.id AS id_solicitud, s.fecha_solicitud,
      v.id_vacacion, v.id_usuario, v.fecha_inicio, v.fecha_fin, v.dias,
      CONCAT_WS(' ', a.Nom, a.Nom2, a.Ape, a.Ape2) AS nombre_funcionario
    FROM vacaciones v
    JOIN solicitudes s ON v.id_solicitud = s.id
    JOIN accounts a    ON v.id_usuario   = a.ID
    WHERE s.tipo_permiso = 'Vacaciones'
      AND s.estado = 'Pendiente'
      AND v.fecha_inicio >= ?
      AND v.fecha_fin    <= ?
      AND a.cod_jefe = ?
      AND a.ID <> ?
  )
  ORDER BY id_solicitud DESC
";

$stmt = $conn->prepare($sql);

/*
 * Tipos de parámetros en orden de aparición:
 *  - Bloque 1: fechaInicio (s), fechaFin (s), N ids (i...), userId (i)
 *  - Bloque 2: fechaInicio (s), fechaFin (s), cod_unidad (i)  <-- si cod_unidad es VARCHAR usa 's'
 *                                             , userId (i)
 *  - Bloque 3: fechaInicio (s), fechaFin (s), cod_jefe (i) (usamos userId), userId (i)
 */
$types  = 'ss' . str_repeat('i', count($ids)) . 'i'   // bloque 1
        . 'ss' . 'i' . 'i'                           // bloque 2  (cambia a 'ss' . 's' . 'i' si cod_unidad es VARCHAR)
        . 'ss' . 'i' . 'i';                          // bloque 3

$params = [];

// Bloque 1
$params[] = $fechaInicio;
$params[] = $fechaFin;
foreach ($ids as $idVal) { $params[] = $idVal; }
$params[] = $userId;

// Bloque 2
$params[] = $fechaInicio;
$params[] = $fechaFin;
$params[] = $unidadJefe;  // si cod_unidad es VARCHAR, asegúrate que sea string
$params[] = $userId;

// Bloque 3
$params[] = $fechaInicio;
$params[] = $fechaFin;
$params[] = $userId;      // legacy: a.cod_jefe = userId
$params[] = $userId;

$stmt->bind_param($types, ...$params);
$stmt->execute();
$result = $stmt->get_result();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Panel Jefe de Departamento - Solicitudes de Vacaciones Pendientes</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        * { margin:0; padding:0; box-sizing:border-box; font-family:'Roboto', sans-serif; }
        body { background:#f0f4f8; color:#2c3e50; line-height:1.6; }
        .container { display:flex; min-height:100vh; }
        .sidebar { width:260px; background:linear-gradient(135deg,#2c3e50,#34495e); color:#fff; padding:20px 15px; position:fixed; height:100%; transition:transform .3s ease; z-index:1000; }
        .sidebar.collapsed { transform:translateX(-260px); width:0; }
        .sidebar h2 { font-size:24px; margin-bottom:30px; text-align:center; color:#68c8d8; }
        .sidebar ul { list-style:none; }
        .sidebar ul li { margin-bottom:15px; }
        .sidebar ul li a { color:#ecf0f1; text-decoration:none; font-size:16px; display:flex; align-items:center; padding:10px 15px; border-radius:5px; transition:background .3s,color .3s; }
        .sidebar ul li a:hover { background:#465c71; color:#68c8d8; }
        .sidebar ul li a i { margin-right:10px; }
        .toggle-btn { display:none; position:fixed; left:10px; top:10px; background:#68c8d8; color:#fff; border:none; padding:10px; border-radius:5px; cursor:pointer; z-index:1001; transition:background .3s; }
        .toggle-btn:hover { background:#5dade2; }
        .main-content { flex-grow:1; padding:30px; margin-left:260px; transition:margin-left .3s ease; width:calc(100% - 260px); }
        .welcome-message { font-size:28px; color:#2c3e50; margin-bottom:25px; text-align:center; font-weight:500; }
        .main-content h1 { font-size:32px; color:#2c3e50; margin-bottom:30px; text-align:center; border-bottom:3px solid #68c8d8; padding-bottom:10px; font-weight:700; }

        .dashboard-grid { display:grid; grid-template-columns:repeat(auto-fit,minmax(320px,1fr)); gap:20px; overflow-x:auto; }
        .solicitud-card { background:#fff; padding:0; border-radius:12px; box-shadow:0 8px 18px rgba(0,0,0,.08); transition:transform .2s, box-shadow .2s; display:flex; flex-direction:column; border:1px solid #e8eef4; }
        .solicitud-card:hover { transform:translateY(-4px); box-shadow:0 12px 24px rgba(0,0,0,.10); }

        .card-header { display:flex; align-items:center; justify-content:space-between; padding:14px 16px; border-bottom:1px solid #eef2f6; background:linear-gradient(180deg,#ffffff 0%,#f7fbff 100%); border-radius:12px 12px 0 0; }
        .title { font-weight:600; color:#2c3e50; font-size:15px; display:flex; align-items:center; gap:8px; }
        .badges { display:flex; gap:6px; flex-wrap:wrap; }
        .badge { font-size:11px; padding:4px 8px; border-radius:999px; background:#eef7ff; color:#1f6fb2; }

        .card-body { padding:16px; display:grid; gap:8px; }
        .row { display:grid; grid-template-columns:150px 1fr; gap:6px; }
        .row .label { color:#6b7c93; font-weight:500; font-size:13px; }
        .row .value { color:#2b3a4a; font-size:14px; word-wrap:break-word; }

        .actions { margin-top:auto; padding:14px; border-top:1px solid #eef2f6; }
        textarea { width:100%; padding:10px; border:1px solid #ddd; border-radius:8px; resize:vertical; font-size:14px; min-height:60px; transition:border-color .3s; }
        textarea:focus { border-color:#68c8d8; outline:none; }
        .button-group { display:flex; gap:8px; flex-wrap:wrap; margin-top:8px; }
        .btn { flex:1; padding:10px; border:none; border-radius:6px; cursor:pointer; font-size:14px; font-weight:600; }
        .btn-approve { background:linear-gradient(135deg,#27ae60,#219653); color:#fff; }
        .btn-reject  { background:linear-gradient(135deg,#c0392b,#a93226); color:#fff; }
        .btn-preview { background:#e9f3ff; color:#0b5cab; border:1px solid #cfe6ff; font-size:12px; padding:6px 10px; flex:0 0 auto; }
        .btn-preview:hover { background:#f3f9ff; }

        /* Modal preview */
        .modal { display:none; position:fixed; inset:0; background:rgba(0,0,0,.5); z-index:2000; align-items:center; justify-content:center; padding:20px; }
        .modal.open { display:flex; }
        .modal-card { width:min(1200px,96vw); height:min(85vh,900px); background:#fff; border-radius:12px; overflow:hidden; box-shadow:0 20px 60px rgba(0,0,0,.25); display:flex; flex-direction:column; }
        .modal-header { display:flex; align-items:center; justify-content:space-between; padding:10px 14px; border-bottom:1px solid #eef2f6; background:#f8fbff; }
        .modal-title { font-weight:600; color:#2c3e50; }
        .modal-close { background:transparent; border:1px solid #e3eaf2; padding:6px 10px; border-radius:6px; cursor:pointer; }
        .modal-iframe { width:100%; height:100%; border:0; }

        @media (max-width:1024px){ .sidebar{width:220px;} .main-content{margin-left:220px;width:calc(100% - 220px);padding:25px;} }
        @media (max-width:768px){ .toggle-btn{display:block;} .sidebar{transform:translateX(-260px);width:260px;} .sidebar.open{transform:translateX(0);} .main-content{margin-left:0;width:100%;padding:20px;} .dashboard-grid{grid-template-columns:1fr;} .solicitud-card{min-height:auto;padding:15px;} .button-group{flex-direction:column;} .btn{width:100%;} .row{grid-template-columns:1fr;} }
        @media (max-width:480px){ .sidebar{padding:10px;} .sidebar h2{font-size:18px;margin-bottom:15px;} .sidebar ul li a{font-size:14px;padding:6px 10px;} .main-content{padding:10px;} .welcome-message{font-size:20px;} .main-content h1{font-size:24px;} }
    </style>
</head>
<body>
    <button class="toggle-btn" onclick="toggleSidebar()"><i class="fas fa-bars"></i></button>
    <div class="container">
        <!-- Menú lateral -->
        <div class="sidebar" id="sidebar">
            <h2>Panel Jefe</h2>
            <ul>
                <li><a href="dashboard_jefe.php"><i class="fas fa-tachometer-alt"></i> Panel Principal</a></li>
                <li><a href="panel_jefe.php"><i class="fas fa-clipboard-list"></i> Solicitudes Pendientes-Permiso</a></li>
                <li><a href="panel_jefe_vacaciones.php"><i class="fas fa-umbrella-beach"></i> Solicitudes Pendientes-Vacaciones</a></li>
                <li><a href="solicitudes_aprobadas.php"><i class="fas fa-check"></i> Solicitudes Aprobadas</a></li>
                <li><a href="solicitudes_rechazadas.php"><i class="fas fa-times"></i> Solicitudes Rechazadas</a></li>
                <li><a href="vacaciones_jefe.php"><i class="fas fa-users"></i> Vacaciones</a></li>
                <li><a href="panel_jefe_user.php"><i class="fas fa-eye"></i> Perfil</a></li>
                <li>
                    <a href="https://sistemas.proasol.gob.hn/denuncias_proasol/index.html?src=jefe_sidebar" target="_blank" rel="noopener">
                       <i class="fas fa-shield-alt"></i> Denuncia-CPEP
                    </a>
                </li>
                <li><a href="logout.php"><i class="fas fa-sign-out-alt"></i> Cerrar Sesión</a></li>
            </ul>
        </div>

        <!-- Contenido principal -->
        <div class="main-content">
            <div class="welcome-message">Bienvenido, <?= htmlspecialchars($_SESSION['Nom'] ?? '') ?></div>
            <h1>Solicitudes de Vacaciones Pendientes - Panel Jefe de Departamento</h1>

            <div class="dashboard-grid">
                <?php if ($result && $result->num_rows > 0): ?>
                    <?php while ($row = $result->fetch_assoc()): ?>
                        <?php
                            $idSolicitud = (int)$row['id_solicitud'];
                            $urlDetalle  = "detalle_solicitud_jefe.php?id=" . $idSolicitud; // Ajusta si usas otro parámetro
                            $nombre      = ucwords(mb_strtolower($row['nombre_funcionario'] ?? '', 'UTF-8'));
                        ?>
                        <div class="solicitud-card">
                            <div class="card-header">
                                <div class="title"><i class="fas fa-user-circle"></i> <?= htmlspecialchars($nombre) ?></div>
                                <div class="badges">
                                    <span class="badge"><i class="fa-regular fa-calendar"></i> Vacaciones</span>
                                    <span class="badge"><i class="fa-regular fa-clock"></i> Pendiente</span>
                                </div>
                            </div>

                            <div class="card-body">
                                <div class="row">
                                    <div class="label">Fecha solicitud:</div>
                                    <div class="value"><?= htmlspecialchars($row['fecha_solicitud'] ?? '') ?></div>
                                </div>
                                <div class="row">
                                    <div class="label">Fecha inicio:</div>
                                    <div class="value"><?= htmlspecialchars($row['fecha_inicio']) ?></div>
                                </div>
                                <div class="row">
                                    <div class="label">Fecha fin:</div>
                                    <div class="value"><?= htmlspecialchars($row['fecha_fin']) ?></div>
                                </div>
                                <div class="row">
                                    <div class="label">Días solicitados:</div>
                                    <div class="value"><?= htmlspecialchars($row['dias']) ?></div>
                                </div>

                                <div class="button-group" style="margin-top:10px;">
                                    <button type="button" class="btn btn-preview" data-url="<?= htmlspecialchars($urlDetalle) ?>">
                                        <i class="fas fa-eye"></i> Ver detalle
                                    </button>
                                </div>
                            </div>

                            <div class="actions">
                                <form action="procesar_revision_vacaciones.php" method="POST">
                                    <input type="hidden" name="id_solicitud" value="<?= htmlspecialchars($idSolicitud) ?>">
                                    <textarea name="comentario" placeholder="Comentario (opcional)"></textarea>
                                    <div class="button-group">
                                        <button type="submit" name="accion" value="Aprobar" class="btn btn-approve">✅ Aprobar</button>
                                        <button type="submit" name="accion" value="Rechazar" class="btn btn-reject">❌ Rechazar</button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    <?php endwhile; ?>
                <?php else: ?>
                    <div class="solicitud-card" style="text-align:center; min-height:200px; justify-content:center; padding:20px;">
                        <p style="color:#777;">No hay solicitudes pendientes.</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Modal de vista previa -->
    <div class="modal" id="previewModal" aria-hidden="true">
        <div class="modal-card">
            <div class="modal-header">
                <div class="modal-title"><i class="fas fa-eye"></i> Vista previa de la solicitud</div>
                <button class="modal-close" id="closePreview" type="button">Cerrar</button>
            </div>
            <iframe id="previewFrame" class="modal-iframe" src="about:blank" title="Vista previa"></iframe>
        </div>
    </div>

    <script>
        function toggleSidebar() {
            const sidebar = document.getElementById('sidebar');
            sidebar.classList.toggle('open');
            const mainContent = document.querySelector('.main-content');
            if (sidebar.classList.contains('open')) {
                mainContent.style.marginLeft = '260px';
                mainContent.style.width = 'calc(100% - 260px)';
            } else {
                mainContent.style.marginLeft = '0';
                mainContent.style.width = '100%';
            }
        }

        // Vista previa (modal)
        (function(){
            const modal = document.getElementById('previewModal');
            const iframe = document.getElementById('previewFrame');
            const closeBtn = document.getElementById('closePreview');

            function openModal(url) {
                iframe.src = url;
                modal.classList.add('open');
                modal.setAttribute('aria-hidden', 'false');
                document.body.style.overflow = 'hidden';
            }

            function closeModal() {
                modal.classList.remove('open');
                modal.setAttribute('aria-hidden', 'true');
                iframe.src = 'about:blank';
                document.body.style.overflow = '';
            }

            document.querySelectorAll('.btn-preview').forEach(btn => {
                btn.addEventListener('click', () => openModal(btn.getAttribute('data-url')));
            });
            closeBtn.addEventListener('click', closeModal);
            modal.addEventListener('click', (e) => { if (e.target === modal) closeModal(); });
            document.addEventListener('keydown', (e) => { if (e.key === 'Escape' && modal.classList.contains('open')) closeModal(); });
        })();
    </script>

    <?php $conn->close(); ?>
</body>
</html>
