<?php
session_start();
require 'conexion.php';

if (!isset($_SESSION['user_id']) || ($_SESSION['role'] != 9 && $_SESSION['role'] != 11)) {
    header("Location: login.php");
    exit;
}

$empleado_id = isset($_GET['id']) ? intval($_GET['id']) : 0;

// --- PROCESAMIENTO DE GUARDADO DE AJUSTES ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['guardar_ajustes'])) {
    $ajustes = json_encode($_POST['periodos']);
    $stmt = $conn->prepare("UPDATE accounts SET ajuste_vacaciones_json = ? WHERE ID = ?");
    $stmt->bind_param("si", $ajustes, $empleado_id);
    $stmt->execute();
    header("Location: ?id=$empleado_id");
    exit;
}

// --- PROCESAMIENTO DE VACACIONES (INSERT/UPDATE) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_vacation'])) {
        $stmt_insert = $conn->prepare("INSERT INTO vacaciones (id_usuario, fecha_inicio, fecha_fin, dias, observaciones) VALUES (?, ?, ?, ?, ?)");
        $stmt_insert->bind_param("issds", $_POST['id_usuario'], $_POST['fecha_inicio'], $_POST['fecha_fin'], $_POST['dias'], $_POST['observaciones']);
        $stmt_insert->execute();
        header("Location: ?id=$empleado_id"); exit;
    }
    if (isset($_POST['edit_vacation'])) {
        $stmt_update = $conn->prepare("UPDATE vacaciones SET fecha_inicio = ?, fecha_fin = ?, dias = ?, observaciones = ? WHERE id_vacacion = ?");
        $stmt_update->bind_param("ssdsi", $_POST['fecha_inicio'], $_POST['fecha_fin'], $_POST['dias'], $_POST['observaciones'], $_POST['id_vacacion']);
        $stmt_update->execute();
        header("Location: ?id=$empleado_id"); exit;
    }
}

// --- OBTENCIÓN DE DATOS ---
$sql_empleados = "SELECT ID, Nom, Nom2, Ape, Ape2 FROM accounts";
$result_empleados = $conn->query($sql_empleados);
$empleados = [];
while ($row = $result_empleados->fetch_assoc()) {
    $empleados[$row['ID']] = trim(implode(' ', array_filter([$row['Nom'], $row['Nom2'], $row['Ape'], $row['Ape2']])));
}

$stmt_empleado = $conn->prepare("SELECT a.*, u.nombre_unidad, p.nombre_puesto FROM accounts a 
    LEFT JOIN unidad u ON a.cod_unidad = u.cod_unidad 
    LEFT JOIN puestos p ON a.cod_puesto = p.cod_puesto WHERE a.ID = ?");
$stmt_empleado->bind_param("i", $empleado_id);
$stmt_empleado->execute();
$empleado = $stmt_empleado->get_result()->fetch_assoc();

$ajustes_guardados = json_decode($empleado['ajuste_vacaciones_json'] ?? '{}', true);
$fecha_ingreso_bd = $empleado['fecha_ingreso'] ?? null;

// --- OBTENCIÓN DE PERMISOS ---
$stmt_permisos = $conn->prepare("SELECT * FROM vacaciones WHERE id_usuario = ? ORDER BY fecha_inicio ASC");
$stmt_permisos->bind_param("i", $empleado_id);
$stmt_permisos->execute();
$result_permisos = $stmt_permisos->get_result();
$permisos = [];
while ($row = $result_permisos->fetch_assoc()) { $permisos[] = $row; }

// --- LÓGICA DE PERIODOS DINÁMICOS POR ANIVERSARIO ---
$periodos_data = [];

if ($fecha_ingreso_bd) {
    $fecha_base = new DateTime($fecha_ingreso_bd);
    $hoy = new DateTime(); // Referencia 2026
    
    // Creamos una copia para iterar año tras año
    $iterador = clone $fecha_base;

    // Generamos periodos mientras el inicio del periodo no supere la fecha actual + 1 año
    while ($iterador <= $hoy) {
        $inicio_p = clone $iterador;
        $fin_p = clone $iterador;
        $fin_p->modify('+1 year')->modify('-1 day'); // Ej: 22/04/2022 al 21/04/2023

        $nombre_p = $inicio_p->format('Y') . " - " . $fin_p->format('Y');
        
        $periodos_data[$nombre_p] = [
            'rango' => $inicio_p->format('d/m/Y') . " al " . $fin_p->format('d/m/Y'),
            'ts_inicio' => $inicio_p->getTimestamp(),
            'ts_fin' => $fin_p->getTimestamp(),
            'items' => [],
            'total_tomados' => 0.0
        ];

        // Clasificar vacaciones dentro de este rango exacto
        foreach ($permisos as $v) {
            $v_inicio_ts = strtotime($v['fecha_inicio']);
            if ($v_inicio_ts >= $inicio_p->getTimestamp() && $v_inicio_ts <= $fin_p->getTimestamp()) {
                $periodos_data[$nombre_p]['items'][] = $v;
                $periodos_data[$nombre_p]['total_tomados'] += (float)$v['dias'];
            }
        }

        $iterador->modify('+1 year');
    }
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Panel de vacaciones RRHH</title>
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <style>
        body { font-family: 'Roboto', sans-serif; background: #f4f7f6; padding: 20px; }
        .container { max-width: 1150px; margin: 0 auto; background: #fff; border-radius: 10px; box-shadow: 0 4px 15px rgba(0,0,0,0.1); overflow: hidden; }
        .header { background: #2c3e50; color: white; padding: 25px; display: flex; justify-content: space-between; align-items: center; }
        .section { padding: 25px; border-bottom: 1px solid #eee; }
        .vacation-table { width: 100%; border-collapse: collapse; margin-top: 15px; }
        .vacation-table th, .vacation-table td { padding: 12px; border: 1px solid #e0e0e0; text-align: left; }
        .vacation-table th { background: #fdfdfd; color: #34495e; }
        .tag-container { display: flex; flex-wrap: wrap; gap: 6px; }
        .vacation-tag { background: #d1ecf1; border: 1px solid #bee5eb; color: #0c5460; padding: 3px 10px; border-radius: 15px; font-size: 11px; font-weight: 500; }
        .rango-fecha { display: block; font-size: 11px; color: #7f8c8d; font-weight: normal; margin-top: 4px; }
        .input-ajuste { width: 90px; padding: 8px; border: 2px solid #3498db; border-radius: 5px; text-align: center; font-weight: bold; }
        .saldo-final { font-weight: 800; color: #c0392b; font-size: 1.2em; }
        .btn-save { background: #27ae60; color: white; border: none; padding: 10px 20px; border-radius: 5px; cursor: pointer; float: right; margin-top: 10px; }
        .add-form { background: #f8f9fa; border: 1px solid #ddd; padding: 20px; display: flex; gap: 10px; align-items: center; border-radius: 8px; }
    </style>
</head>
<body>

<div class="container">
    <div class="header">
        <div>
            <h1 style="margin:0;">Panel de vacaciones RRHH</h1>
            <small>Ingreso: <strong><?= $fecha_ingreso_bd ? date('d/m/Y', strtotime($fecha_ingreso_bd)) : 'No registrada' ?></strong></small>
        </div>
        <select class="js-example-basic-single" id="employee-select" style="width: 350px;">
            <?php foreach ($empleados as $id => $nombre): ?>
                <option value="<?= $id ?>" <?= $id == $empleado_id ? 'selected' : '' ?>><?= htmlspecialchars($nombre) ?></option>
            <?php endforeach; ?>
        </select>
    </div>

    <div class="section">
        <form method="POST">
            <input type="hidden" name="guardar_ajustes" value="1">
            <table class="vacation-table">
                <thead>
                    <tr>
                        <th width="20%">Periodo Aniversario</th>
                        <th width="35%">Días Tomados (Detalle)</th>
                        <th width="15%">Total Tomados (A)</th>
                        <th width="15%">Días Asignados (B)</th>
                        <th width="15%">Saldo Pendiente (B-A)</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($periodos_data)): ?>
                        <tr><td colspan="5" style="text-align:center;">Seleccione un empleado con fecha de ingreso.</td></tr>
                    <?php else: ?>
                        <?php foreach (array_reverse($periodos_data) as $nombre => $data): 
                            $val_asignado = $ajustes_guardados[$nombre] ?? 0.0;
                            $saldo = $val_asignado - $data['total_tomados'];
                        ?>
                        <tr>
                            <td>
                                <strong><?= $nombre ?></strong>
                                <span class="rango-fecha"><?= $data['rango'] ?></span>
                            </td>
                            <td>
                                <div class="tag-container">
                                    <?php if(empty($data['items'])): ?>
                                        <span style="color:#ccc; font-size:11px;">Sin vacaciones</span>
                                    <?php endif; ?>
                                    <?php foreach ($data['items'] as $item): ?>
                                        <span class="vacation-tag" title="<?= htmlspecialchars($item['observaciones']) ?>">
                                            <?= date('d/m/y', strtotime($item['fecha_inicio'])) ?> (<?= $item['dias'] ?>)
                                        </span>
                                    <?php endforeach; ?>
                                </div>
                            </td>
                            <td><span class="total-tomados"><?= number_format($data['total_tomados'], 1) ?></span></td>
                            <td>
                                <input type="number" step="0.5" name="periodos[<?= $nombre ?>]" 
                                       class="input-ajuste" value="<?= $val_asignado ?>" 
                                       oninput="calcularSaldo(this)">
                            </td>
                            <td><span class="saldo-final"><?= number_format($saldo, 1) ?></span></td>
                        </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
            <button type="submit" class="btn-save">Guardar Cambios</button>
        </form>
    </div>

    <div class="section">
        <h3>Historial Detallado</h3>
        <table class="vacation-table">
            <?php foreach ($permisos as $p): ?>
            <tr>
                <form method="POST">
                    <input type="hidden" name="id_vacacion" value="<?= $p['id_vacacion'] ?>">
                    <td><input type="date" name="fecha_inicio" value="<?= $p['fecha_inicio'] ?>"></td>
                    <td><input type="date" name="fecha_fin" value="<?= $p['fecha_fin'] ?>"></td>
                    <td><input type="number" step="0.5" name="dias" value="<?= $p['dias'] ?>" style="width:60px;"></td>
                    <td><input type="text" name="observaciones" value="<?= htmlspecialchars($p['observaciones']) ?>" style="width:100%;"></td>
                    <td><button type="submit" name="edit_vacation">💾</button></td>
                </form>
            </tr>
            <?php endforeach; ?>
        </table>

        <h4>Nuevo Registro</h4>
        <form class="add-form" method="POST">
            <input type="hidden" name="id_usuario" value="<?= $empleado_id ?>">
            <input type="date" name="fecha_inicio" required>
            <input type="date" name="fecha_fin" required>
            <input type="number" step="0.5" name="dias" placeholder="Días" style="width:80px;" required>
            <input type="text" name="observaciones" placeholder="Observaciones..." style="flex:1;">
            <button type="submit" name="add_vacation" style="background:#2c3e50; color:white; padding:10px 20px; border:none; border-radius:5px; cursor:pointer;">+ Agregar</button>
        </form>
    </div>
</div>

<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
<script>
    $(document).ready(function() {
        $('.js-example-basic-single').select2();
        $('#employee-select').on('change', function() {
            window.location.href = '?id=' + this.value;
        });
    });

    function calcularSaldo(inputElement) {
        const fila = inputElement.closest('tr');
        const tomados = parseFloat(fila.querySelector('.total-tomados').innerText) || 0;
        const asignados = parseFloat(inputElement.value) || 0;
        fila.querySelector('.saldo-final').innerText = (asignados - tomados).toFixed(1);
    }
</script>
</body>
</html>