<?php
session_start();
require 'conexion.php';

// Verificar que el usuario esté autenticado
if (!isset($_SESSION['user_id'])) {
    header("Location: index.php");
    exit;
}

// ✅ Función para redirigir según rol
function obtenerDashboardPorRol($role) {
    switch ($role) {
        case 7: return "dashboard_usuario.php";
        case 8: return "dashboard_jefe.php";
        case 9:
        case 11: return "dashboard_rrhh.php";
        case 12: return "dashboard_directora.php";
        default: return "login.php";
    }
}

$user_id = $_SESSION['user_id'];
$tipo_contrato = null; // almacenará el tipo de contrato del usuario

// Obtener los datos del usuario (incluyendo tipo_contrato)
$sql_usuario = "SELECT Nom, Nom2, Ape, Ape2, cod_unidad, cod_jefe, cod_puesto, tipo_contrato FROM accounts WHERE ID = ?";
if ($stmt = $conn->prepare($sql_usuario)) {
    $stmt->bind_param('i', $user_id);
    $stmt->execute();
    $stmt->bind_result($nom, $nom2, $ape, $ape2, $cod_unidad, $cod_jefe, $cod_puesto, $tipo_contrato);
    $stmt->fetch();
    $nombre_completo = trim(implode(' ', array_filter([$nom, $nom2, $ape, $ape2])));
    $_SESSION['nombre_completo'] = $nombre_completo;
    $stmt->close();

    function fetchCampo($conn, $tabla, $campo, $cod, $alias) {
        $sql = "SELECT $campo FROM $tabla WHERE $alias = ?";
        $st = $conn->prepare($sql);
        $st->bind_param('i', $cod);
        $st->execute();
        $st->bind_result($val);
        $st->fetch();
        $st->close();
        return $val;
    }

    $_SESSION['unidad'] = $cod_unidad
        ? (fetchCampo($conn, 'unidad', 'nombre_unidad', $cod_unidad, 'cod_unidad') ?: 'Unidad no encontrada')
        : 'Unidad no asignada';
    $_SESSION['puesto'] = $cod_puesto
        ? (fetchCampo($conn, 'puestos', 'nombre_puesto', $cod_puesto, 'cod_puesto') ?: 'Puesto no encontrado')
        : 'Puesto no asignada';
    $_SESSION['nombre_jefe_inmediato'] = $cod_jefe
        ? (fetchCampo($conn, 'jefes', 'nombre_jefe', $cod_jefe, 'cod_jefe') ?: 'Jefe no encontrado')
        : 'Jefe no asignado';
} else {
    $_SESSION['nombre_completo'] = "Error al obtener el nombre";
    $_SESSION['unidad'] = 'Error al obtener la unidad';
    $_SESSION['nombre_jefe_inmediato'] = 'Error al obtener el jefe';
}

date_default_timezone_set('America/Tegucigalpa');
$fecha_solicitud = date('Y-m-d');

/* ============================
   📌 Cálculo SERVER-SIDE para la tarjeta de disponibilidad
   Excluye EXACTAMENTE 2025-09-29 → 2025-09-30
   ============================ */

// 1️⃣ Límite base según tipo de contrato
if ($tipo_contrato === 'Contrato') {
    $LIMITE_VAC = 6.0;   // Contrato → 6 días
} else {
    $LIMITE_VAC = 12.0;  // Resto → 12 días
}

// 2️⃣ Overrides por ID específicos
// IDs con 5 días
$ids_5_dias = [165, 34, 47, 41, 71, 81, 131, 83];
// IDs con 6 días
$ids_6_dias = [166, 167, 48];
// IDs con 4 días
$ids_4_dias = [168, 164];
// ID con 4.5 días
$id_4_5_dias = 162;

if (in_array($user_id, $ids_5_dias, true)) {
    $LIMITE_VAC = 5.0;
} elseif (in_array($user_id, $ids_6_dias, true)) {
    $LIMITE_VAC = 6.0;
} elseif (in_array($user_id, $ids_4_dias, true)) {
    $LIMITE_VAC = 4.0;
} elseif ($user_id == $id_4_5_dias) {
    $LIMITE_VAC = 4.5;
}

$dias_usados_ex = 0.0;

$stmt_vac = $conn->prepare("
    SELECT
      ROUND(SUM(
        CASE
          -- ❌ Excluir rango exacto
          WHEN DATE(v.fecha_inicio) = '2025-09-29'
           AND DATE(v.fecha_fin)    = '2025-09-30' THEN 0

          -- ❌ Excluir fechas individuales
          WHEN DATE(v.fecha_inicio) IN (
                '2025-12-24',
                '2025-12-26',
                '2025-12-31',
                '2026-01-02'
          )
          OR   DATE(v.fecha_fin) IN (
                '2025-12-24',
                '2025-12-26',
                '2025-12-31',
                '2026-01-02'
          ) THEN 0

          -- ✅ Caso normal
          ELSE v.dias
        END
      ), 1) AS usados_ex
    FROM vacaciones v
    LEFT JOIN solicitudes s ON v.id_solicitud = s.id
    WHERE v.id_usuario = ?
      AND (
            v.id_solicitud IS NULL
         OR (s.estado = 'Aprobada por Recursos Humanos'
             AND v.fecha_inicio >= '2025-07-01')
      )
");

$stmt_vac->bind_param('i', $user_id);
$stmt_vac->execute();
$stmt_vac->bind_result($usados_tmp);
$stmt_vac->fetch();
$stmt_vac->close();

$dias_usados_ex = (float)($usados_tmp ?? 0);
$dias_restantes = max(0, round($LIMITE_VAC - $dias_usados_ex, 1));
$pct_uso = ($LIMITE_VAC > 0) ? max(0, min(100, round(($dias_usados_ex / $LIMITE_VAC) * 100))) : 0;

// Helper para imprimir sin .0
function fmt_num($n) {
    return (floor($n) == $n) ? (string)intval($n) : number_format($n, 1, '.', '');
}

// 🔵 Datos autoritativos para la validación “Pendiente + límite alcanzado”
$anioActual = (int)date('Y');

// ¿Existe alguna solicitud de Vacaciones en estado Pendiente?
$sql_has_pending = "SELECT EXISTS(
    SELECT 1 FROM solicitudes
    WHERE id_usuario = ? AND tipo_permiso = 'Vacaciones' AND estado = 'Pendiente'
) AS has_pending";
$stmtP = $conn->prepare($sql_has_pending);
$stmtP->bind_param('i', $user_id);
$stmtP->execute();
$resP = $stmtP->get_result()->fetch_assoc();
$stmtP->close();
$tiene_pendiente = (int)($resP['has_pending'] ?? 0);

// Suma de días del año en vacaciones
$sql_total_anio = "SELECT COALESCE(SUM(dias), 0) AS total_dias
                   FROM vacaciones
                   WHERE id_usuario = ? AND YEAR(fecha_inicio) = ?";
$stmtT = $conn->prepare($sql_total_anio);
$stmtT->bind_param('ii', $user_id, $anioActual);
$stmtT->execute();
$stmtT->bind_result($total_anio);
$stmtT->fetch();
$stmtT->close();

$conn->close();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Solicitud de Vacaciones</title>
    <script src="https://cdn.jsdelivr.net/npm/signature_pad@4.0.0/dist/signature_pad.umd.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <style>
        body { font-family: Arial, sans-serif; background-color: #f5f5f5; margin: 0; padding: 0; }
        .container { max-width: 800px; margin: 20px auto; background-color: white; padding: 20px; border: 1px solid #ccc; box-shadow: 0 0 10px rgba(0,0,0,0.1); position: relative; }
        .ribbon { position: absolute; top: 0; left: 0; width: 10px; height: 100%; background-color: #68c8d8; }
        header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; }
        header h1 { font-size: 24px; margin: 0; color: #333; }
        .logo-left img { width: 100px; height: 100px; }
        .logo-right img { width: 150px; height: 100px; }
        .form-group { margin-bottom: 15px; }
        .form-group label { display: block; font-weight: bold; margin-bottom: 5px; }
        .form-group input, .form-group textarea, .form-group select { width: 100%; padding: 8px; border: 1px solid #ccc; border-radius: 4px; box-sizing: border-box; }
        .form-group textarea { resize: vertical; }
        .form-group.inline { display: flex; gap: 20px; }
        .form-group.inline > div { flex: 1; }
        .form-group.inline .motivo input { width: 150px; }
        .form-group.inline .date input { width: 150px; }
        .date-group { flex: 1; }
        .input-row { display: flex; gap: 10px; align-items: center; }
        .input-row input[type="date"], .input-row select { flex: 1; padding: 8px; border: 1px solid #ccc; border-radius: 4px; }
        .signature-line { border-bottom: 1px solid #000; height: 20px; margin-top: 10px; }
        .signatures { margin-top: 30px; }
        .jefe { text-align: center; margin-top: 30px; margin-left: auto; margin-right: auto; }
        .jefe p { margin: 5px 0; font-weight: bold; }
        .observaciones h3 { margin-bottom: 10px; font-size: 16px; }
        .observaciones ul { list-style-type: disc; padding-left: 20px; }
        .observaciones li { margin-bottom: 10px; }
        footer { text-align: center; margin-top: 30px; font-size: 12px; color: #666; }
        footer p { margin: 5px 0; }
        .save-button { display: block; margin: 20px auto; padding: 10px 20px; background-color: #28a745; color: white; border: none; border-radius: 5px; font-size: 16px; cursor: pointer; transition: background 0.3s; }
        .save-button:hover { background-color: #218838; }
        .approval-message { text-align: center; margin: 20px auto; padding: 10px 20px; background-color: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; border-radius: 5px; font-size: 16px; }
        .signature-pad { border: 1px solid #ccc; height: 150px; width: 350px; margin-top: 10px; }
        .clear-button { margin-top: 10px; padding: 5px 10px; background-color: #dc3545; color: white; border: none; border-radius: 5px; cursor: pointer; transition: background 0.3s; }
        .clear-button:hover { background-color: #c82333; }

        /* PRINT */
        @media print {
            html, body { margin: 0 !important; padding: 0 !important; height: 100%; width: 100%; background: none !important; -webkit-print-color-adjust: exact !important; print-color-adjust: exact !important; }
            @page { size: A4; margin: 5mm; }
            .container { width: 200mm; min-height: 287mm; margin: 0; padding: 10mm; background-color: white !important; border: 1px solid #ccc !important; box-shadow: none; position: relative; display: flex; flex-direction: column; justify-content: space-between; }
            .ribbon { position: absolute; top: 0; left: 0; width: 10px; height: 100%; background-color: #68c8d8 !important; }
            .signature-pad { width: 350px !important; height: 150px !important; }
            footer { text-align: center; margin-top: 30px; font-size: 10px; color: #666 !important; page-break-inside: avoid; }
            footer p { margin: 6px 0; }
            .save-button, .approval-message, .clear-button { display: none !important; }
        }

        /* Responsive */
        @media (max-width: 900px) {
            .container { max-width: 98vw; padding: 10px; }
            header { flex-direction: column; align-items: flex-start; gap: 10px; }
            .logo-left img, .logo-right img { width: 80px; height: auto; }
        }
        @media (max-width: 600px) {
            .container { max-width: 100vw; padding: 5px; border: none; box-shadow: none; }
            header { flex-direction: column; align-items: center; gap: 10px; }
            .logo-left img, .logo-right img { width: 60px; height: auto; }
            .form-group.inline { flex-direction: column; gap: 0; }
            .form-group.inline > div { width: 100%; }
            .signature-pad { width: 100% !important; min-width: 200px; max-width: 100vw; }
            .save-button { width: 100%; font-size: 18px; }
            .form-group.inline .datetime { width: 100%; }
        }
        @media (max-width: 600px) { .form-group { padding-left: 10px; } }

        /* 🔙 Botón Atrás flotante */
        .back-button {
            position: fixed; bottom: 20px; left: 20px;
            background: linear-gradient(135deg, #e74c3c, #c0392b); color: white;
            padding: 12px 20px; border-radius: 8px; font-size: 15px; font-weight: 600; text-decoration: none;
            box-shadow: 0 4px 8px rgba(0,0,0,0.2); transition: background 0.3s, transform 0.2s; z-index: 999;
        }
        .back-button:hover { background: linear-gradient(135deg, #c0392b, #e74c3c); transform: translateY(-2px); }

        /* Efectos para la tarjeta de disponibilidad */
        #vacaciones-card.vac-danger { background-color: #fff0f0 !important; border-color: #e74c3c !important; box-shadow: 0 0 0 2px rgba(231,76,60,.25) inset; }
        @keyframes vacPulse {
          0%, 100% { box-shadow: 0 0 0 0 rgba(231,76,60,.5), 0 0 0 0 rgba(231,76,60,.3) inset; }
          50% { box-shadow: 0 0 0 6px rgba(231,76,60,.15), 0 0 0 2px rgba(231,76,60,.35) inset; background-color: #ffe8e8; }
        }
        #vacaciones-card.vac-flash { animation: vacPulse 0.9s ease-in-out infinite; background-color: #ffeaea !important; border-color: #e74c3c !important; }
        #vacaciones-card.vac-flash #vac-progress { background: #e74c3c !important; }
        #vacaciones-card.vac-flash #vac-msg { color: #a61b1b !important; }

        /* 🔵 NUEVO: feedback visual de botón deshabilitado */
        .save-button[disabled] { opacity: .6; cursor: not-allowed; }
    </style>
</head>
<body>
<div class="container">
    <div class="ribbon"></div>
    <header>
        <div class="logo-left"><img src="logo-proasol.png" alt="Logo PROASOL"></div>
        <h1>Solicitud de Vacaciones</h1>
        <div class="logo-right"><img src="gobierno-logo.png" alt="Logo HONDUR"></div>
    </header>

    <!-- 🔙 Botón Atrás -->
    <a href="<?= obtenerDashboardPorRol($_SESSION['role'] ?? 0) ?>" class="back-button">⬅ Volver al Panel</a>

    <!-- Tarjeta de disponibilidad con valores del SERVIDOR -->
    <div class="form-group" id="vacaciones-card" style="border:1px solid #e3e3e3;border-radius:6px;padding:10px;background:#fafafa;margin-bottom:15px;">
      <label style="font-weight:bold;">Disponibilidad de Vacaciones <small>(Julio 2024 - Julio 2025)</small></label>
      <div class="input-row" style="justify-content:space-between;">
        <div>Usados: <strong><span id="vac-used"><?= fmt_num($dias_usados_ex) ?></span></strong> /
            Límite: <strong><span id="vac-limit"><?= fmt_num($LIMITE_VAC) ?></span></strong></div>
        <div>Disponibles: <strong><span id="vac-remaining"><?= fmt_num($dias_restantes) ?></span></strong> días</div>
      </div>
      <div style="height:8px;background:#eee;border-radius:4px;overflow:hidden;margin-top:6px;">
        <div id="vac-progress" style="height:8px;width:<?= $pct_uso ?>%;background:<?= $dias_restantes <= 0 ? '#e74c3c' : '#68c8d8' ?>;"></div>
      </div>
      <small id="vac-msg" style="color:#666">
        <?= ($dias_restantes > 0)
            ? "Te quedan " . fmt_num($dias_restantes) . " día(s) disponibles."
            : "Has alcanzado el límite de " . fmt_num($LIMITE_VAC) . " día(s)." ?>
      </small>
    </div>

    <form id="solicitudForm" action="procesar_solicitud_vacaciones.php" method="POST">
        <div class="form-group">
            <label for="nombre-funcionario">Nombre del Funcionario</label>
            <input type="text" id="nombre-funcionario" name="nombre-funcionario"
                   value="<?= htmlspecialchars($_SESSION['nombre_completo'] ?? '') ?>" required readonly>
        </div>

        <div class="form-group inline">
            <div>
                <label for="unidad">Unidad</label>
                <input type="text" id="unidad" name="unidad" value="<?= htmlspecialchars($_SESSION['unidad'] ?? '') ?>" required readonly>
            </div>
            <div>
                <label for="puesto">Puesto</label>
                <input type="text" id="puesto" name="puesto" value="<?= htmlspecialchars($_SESSION['puesto'] ?? '') ?>" required readonly>
            </div>
        </div>

        <div class="form-group">
            <label for="nombre-jefe">Nombre del Jefe Inmediato</label>
            <input type="text" id="nombre-jefe" name="nombre-jefe" value="<?= htmlspecialchars($_SESSION['nombre_jefe_inmediato'] ?? '') ?>" required readonly>
        </div>

        <div class="form-group inline">
            <div>
                <label for="fecha-solicitud">Fecha de Solicitud</label>
                <input type="date" id="fecha-solicitud" name="fecha-solicitud" value="<?= $fecha_solicitud ?>">
            </div>
            <div class="motivo">
                <label for="motivo">Motivo del Permiso</label>
                <input type="text" id="motivo" name="motivo" value="Vacaciones" readonly>
            </div>
        </div>

        <div class="form-group inline">
            <div class="date-group">
                <label for="fecha-inicial">Seleccione la Fecha de Inicio</label>
                <div class="input-row">
                    <input type="date" id="fecha-inicial" name="fecha-inicial" required>
                    <select id="hora-inicio" name="hora-inicio">
                        <option value="1">Tomar Día Completo</option>
                        <option value="0.5">Tomar medio dia por la Mañana</option>
                        <option value="0.5">Tomar medio dia por la Tarde</option>
                    </select>
                </div>
            </div>
            <div class="date-group">
                <label for="fecha-final">Seleccione la Fecha de Fin</label>
                <div class="input-row">
                    <input type="date" id="fecha-final" name="fecha-final" required>
                    <select id="hora-fin" name="hora-fin">
                        <option value="1">Tomar Día Completo</option>
                        <option value="0.5">Tomar medio dia por la Mañana</option>
                    </select>
                </div>
            </div>
        </div>

        <!-- Campo oculto para días y medio días -->
        <input type="hidden" id="horario" name="horario">

        <!-- Mostrar resultado -->
        <div class="form-group">
            <p><strong>Días Restados de Vacaciones:</strong> <span id="dias-calculados">0</span></p>
        </div>

        <footer>
            <p>CENTRO CÍVICO GUBERNAMENTAL, TORRE 2, PISO 1, BOULEVARD JUAN PABLO II, TEGUCIGALPA, M.D.C., HONDURAS, C.A.</p>
            <p>(+504) 2242-7981</p>
        </footer>
        <button type="submit" class="save-button">Enviar</button>
    </form>
    <p class="approval-message">Podrá imprimir su permiso cuando RRHH lo haya aprobado</p>
</div>

<script>
/* ========= utilidades para fechas mínimas ========= */
function getFechaMinima() {
    const date = new Date();
    let diasHabilesContados = 0;
    while (diasHabilesContados < 3) {
        date.setDate(date.getDate() - 1);
        const day = date.getDay();
        if (day !== 0 && day !== 6) diasHabilesContados++;
    }
    const pad = n => n.toString().padStart(2, '0');
    return `${date.getFullYear()}-${pad(date.getMonth() + 1)}-${pad(date.getDate())}`;
}
window.addEventListener('DOMContentLoaded', () => {
    const minFecha = getFechaMinima();
    document.getElementById('fecha-inicial').setAttribute('min', minFecha);
    document.getElementById('fecha-final').setAttribute('min', minFecha);
});
</script>

<script>
// Cálculo de días hábiles con fracciones
function getWorkingDays(startDate, endDate, startFraction, endFraction) {
    const start = new Date(startDate);
    const end = new Date(endDate);
    start.setMinutes(start.getMinutes() + start.getTimezoneOffset());
    end.setMinutes(end.getMinutes() + end.getTimezoneOffset());
    if (end < start) return 0;

    const sF = parseFloat(startFraction);
    const eF = parseFloat(endFraction);

    if (start.getTime() === end.getTime()) {
        const day = start.getDay();
        return (day !== 0 && day !== 6) ? Math.max(sF, eF) : 0;
    }

    let totalDays = 0;
    const startDay = start.getDay();
    const endDay = end.getDay();

    if (startDay !== 0 && startDay !== 6) totalDays += sF;
    if (endDay !== 0 && endDay !== 6) totalDays += eF;

    const current = new Date(start);
    current.setDate(current.getDate() + 1);
    while (current < end) {
        const day = current.getDay();
        if (day !== 0 && day !== 6) totalDays++;
        current.setDate(current.getDate() + 1);
    }
    return totalDays;
}

// === Datos del servidor para la tarjeta (EXCLUYE 29-09-2025 → 30-09-2025) ===
const VAC_INFO = {
  limit: <?= json_encode($LIMITE_VAC) ?>,
  used: <?= json_encode($dias_usados_ex) ?>,
  remaining: <?= json_encode($dias_restantes) ?>
};

// Estado autoritativo para la regla “Pendiente + límite alcanzado”
const VAC_STATUS = {
  hasPending: <?= json_encode((bool)$tiene_pendiente) ?>,
  totalYear: <?= json_encode((float)$total_anio) ?>,
  limit: <?= json_encode((float)$LIMITE_VAC) ?>
};

// ID del usuario para control de solicitudes duplicadas
const CURRENT_USER_ID = <?= (int)$user_id ?>;

// Presentación/validación en la tarjeta
function updateDisponibilidadUI(info, solicitados){
  const usedEl = document.getElementById('vac-used');
  const limitEl = document.getElementById('vac-limit');
  const remEl = document.getElementById('vac-remaining');
  const prog  = document.getElementById('vac-progress');
  const msg   = document.getElementById('vac-msg');
  const card  = document.getElementById('vacaciones-card');

  const limit = (typeof info?.limit === 'number') ? info.limit : 12;
  const used  = (typeof info?.used  === 'number') ? info.used  : 0;
  const remaining = Math.max(0, (typeof info?.remaining === 'number') ? info.remaining : (limit - used));

  if (usedEl)  usedEl.textContent  = (Math.round(used * 10) / 10).toString().replace('.0','');
  if (limitEl) limitEl.textContent = (Math.round(limit * 10) / 10).toString().replace('.0','');
  if (remEl)   remEl.textContent   = (Math.round(remaining * 10) / 10).toString().replace('.0','');

  if (prog) {
    const pct = limit > 0 ? Math.min(100, Math.round((used / limit) * 100)) : 0;
    prog.style.width = pct + '%';
    prog.style.background = remaining <= 0 ? '#e74c3c' : '#68c8d8';
  }

  const excede = (remaining <= 0) || (solicitados > remaining && solicitados > 0);
  if (msg) {
    if (remaining <= 0) {
      msg.textContent = 'No tienes días disponibles en este periodo (Julio 2024 – Julio 2025).';
    } else if (solicitados > remaining) {
      const total = (used + solicitados).toFixed(1);
      msg.textContent = `Tu selección supera el límite de ${limit} días. Llevas ${used.toFixed(1)}, solicitas ${solicitados.toFixed(1)} → total ${total}.`;
    } else if (solicitados > 0) {
      msg.textContent = `Vas a solicitar ${solicitados.toFixed(1)} día(s). Te quedarían ${(remaining - solicitados).toFixed(1)}.`;
    } else {
      msg.textContent = 'Selecciona fechas/medios días para validar.';
    }
  }

  if (card) {
    card.classList.toggle('vac-danger', excede);
    card.classList.toggle('vac-flash',  excede);
  }
  const submitBtn = document.querySelector('#solicitudForm button[type="submit"].save-button');
  if (submitBtn) submitBtn.disabled = !!excede;
}

// Inicializar tarjeta y reaccionar a cambios del formulario
document.addEventListener('DOMContentLoaded', () => {
  // Inicial
  updateDisponibilidadUI(VAC_INFO, 0);

  // Si hay Pendiente y totalYear >= limit, deshabilitar envío al cargar
  if (VAC_STATUS.hasPending && VAC_STATUS.totalYear >= VAC_STATUS.limit) {
    const btn = document.querySelector('#solicitudForm button[type="submit"].save-button');
    if (btn) btn.disabled = true;
  }

  const recalc = () => {
    const ini = document.getElementById('fecha-inicial').value;
    const fin = document.getElementById('fecha-final').value;
    const horaIni = document.getElementById('hora-inicio').value;
    const horaFin = document.getElementById('hora-fin').value;
    const dias = (ini && fin && new Date(fin) >= new Date(ini)) ? getWorkingDays(ini, fin, horaIni, horaFin) : 0;
    document.getElementById('dias-calculados').textContent = dias.toFixed(1);
    document.getElementById('horario').value = dias.toFixed(1);
    updateDisponibilidadUI(VAC_INFO, dias);
  };

  ['fecha-inicial','fecha-final','hora-inicio','hora-fin'].forEach(id=>{
    const el = document.getElementById(id);
    if (el) el.addEventListener('change', recalc);
  });
});

// Validación de envío
document.getElementById('solicitudForm').addEventListener('submit', function (ev) {

  // 🔒 1) Bloqueo de duplicado por misma fecha de inicio en el mismo día
 const iniFecha = document.getElementById('fecha-inicial').value;

if (iniFecha) {

    // Eliminado: cálculo de fecha actual
    // Eliminado: creación de la llave única
    // Eliminado: validación con localStorage
    // Eliminado: localStorage.setItem(...)

    // Ahora este bloque NO bloquea nada.
    // La validación real la hará el servidor (PHP).
    
    // Si deseas dejar una validación opcional, la puedes colocar aquí.
}


  // 🔒 2) Bloqueo por “Pendiente + límite alcanzado”
  if (VAC_STATUS.hasPending && VAC_STATUS.totalYear >= VAC_STATUS.limit) {
    ev.preventDefault();
    Swal.fire({
      icon:'warning',
      title:'No se puede enviar',
      text:`Tienes vacaciones pendientes de aprobación y ya alcanzaste el límite de ${VAC_STATUS.limit} día(s).`
    });
    return;
  }

  const ini = document.getElementById('fecha-inicial').value;
  const fin = document.getElementById('fecha-final').value;
  const horaIni = document.getElementById('hora-inicio').value;
  const horaFin = document.getElementById('hora-fin').value;

  const dias = (ini && fin && new Date(fin) >= new Date(ini)) ? getWorkingDays(ini, fin, horaIni, horaFin) : 0;
  document.getElementById('horario').value = dias.toFixed(1);

  if (dias > (VAC_INFO.remaining || 0)) {
    ev.preventDefault();
    Swal.fire({
      icon:'warning',
      title:'Excede el disponible',
      text:`Estás solicitando ${dias.toFixed(1)} día(s) pero solo tienes ${(VAC_INFO.remaining || 0).toFixed(1)} disponible(s).`
    });
    return;
  }
}, true);
</script>

<script>
// Sincronizar select de hora fin cuando es mismo día
function toggleHoraFinSelect() {
  const fechaInicio = document.getElementById('fecha-inicial').value;
  const fechaFin = document.getElementById('fecha-final').value;
  const horaInicio = document.getElementById('hora-inicio').value;
  const horaFin = document.getElementById('hora-fin');

  if (fechaInicio && fechaFin && fechaInicio === fechaFin) {
      horaFin.value = horaInicio;
      horaFin.disabled = true;
  } else {
      horaFin.disabled = false;
  }
}
['fecha-inicial','fecha-final','hora-inicio'].forEach(id=>{
  const el = document.getElementById(id);
  if (el) el.addEventListener('change', () => { toggleHoraFinSelect(); });
});
</script>

<script>
// Mensajes de sesión (opcional)
<?php if (isset($_SESSION['success_message'])): ?>
Swal.fire({ icon:'success', title:'¡Éxito!', text:'<?= htmlspecialchars($_SESSION['success_message']); ?>', confirmButtonText:'Aceptar', confirmButtonColor:'#28a745' });
<?php unset($_SESSION['success_message']); endif; ?>

<?php if (isset($_SESSION['error_message'])): ?>
Swal.fire({ icon:'error', title:'Error', text:'<?= htmlspecialchars($_SESSION['error_message']); ?>', confirmButtonText:'Aceptar', confirmButtonColor:'#dc3545' });
<?php unset($_SESSION['error_message']); endif; ?>
</script>
</body>
</html>
