<?php
// procesar_pase_vacaciones.php.php
session_start();
require 'conexion.php';
require 'email_config.php';

if (!isset($_SESSION['user_id'])) {
    die("Acceso denegado");
}

$id_usuario = (int)$_SESSION['user_id'];
$unidad = $_POST['unidad'] ?? '';
$puesto = $_POST['puesto'] ?? '';
$nombre_jefe_inmediato = $_POST['nombre-jefe'] ?? '';
$fecha_solicitud = $_POST['fecha-solicitud'] ?? date('Y-m-d');
$fecha_salida = $_POST['fecha-inicial'] ?? null;
$fecha_fin = $_POST['fecha-final'] ?? null;
$tipo_permiso = 'Vacaciones';
$descripcion_motivo = 'Vacaciones';
$estado = 'Pendiente';
$LIMITE_VACACIONES_ANUAL = 12.0;

// Validaciones de fechas
if (empty($fecha_salida) || empty($fecha_fin)) {
    $_SESSION['error_message'] = 'Por favor, complete las fechas.';
    header("Location: pase_vacaciones.php.php");
    exit;
}
if (strtotime($fecha_fin) < strtotime($fecha_salida)) {
    $_SESSION['error_message'] = 'La fecha final no puede ser anterior a la fecha inicial.';
    header("Location: pase_vacaciones.php.php");
    exit;
}

// Cálculo de días hábiles (incluye extremos L-V)
function getWorkingDays($startDate, $endDate) {
    $start = new DateTime($startDate);
    $end = new DateTime($endDate);
    $workingDays = 0;
    while ($start <= $end) {
        if ((int)$start->format('N') <= 5) { // 1..5 = L-V
            $workingDays++;
        }
        $start->modify('+1 day');
    }
    return $workingDays;
}

// Días solicitados: si viene "horario" numérico lo respetamos; si no, calculamos hábiles
if (isset($_POST['horario']) && is_numeric($_POST['horario'])) {
    $dias_solicitados = (float)$_POST['horario'];
} else {
    $dias_solicitados = (float)getWorkingDays($fecha_salida, $fecha_fin);
}

if ($dias_solicitados <= 0) {
    $_SESSION['error_message'] = 'Los días solicitados deben ser mayores que 0.';
    header("Location: pase_vacaciones.php.php");
    exit;
}

// Obtener username (firma)
$sql_username = "SELECT username FROM accounts WHERE ID = ?";
$stmt_username = $conn->prepare($sql_username);
$stmt_username->bind_param('i', $id_usuario);
$stmt_username->execute();
$result_username = $stmt_username->get_result();
$row_username = $result_username->fetch_assoc();
$stmt_username->close();
$firma_funcionario = $row_username['username'] ?? '---';

// Nombre completo para quien firma
$sql_nombre = "SELECT Nom, Nom2, Ape, Ape2 FROM accounts WHERE ID = ?";
$stmt_nombre = $conn->prepare($sql_nombre);
$stmt_nombre->bind_param('i', $id_usuario);
$stmt_nombre->execute();
$stmt_nombre->bind_result($nom, $nom2, $ape, $ape2);
$stmt_nombre->fetch();
$nombre_funcionario_firma = trim(implode(' ', array_filter([$nom, $nom2, $ape, $ape2])));
$stmt_nombre->close();

// Transacción
$conn->begin_transaction();
try {
    $anio = (int)date('Y', strtotime($fecha_salida));
// 🔵 NUEVO: ¿existe ya una solicitud PENDIENTE?
$sql_has_pending = "SELECT EXISTS(
    SELECT 1 FROM solicitudes
    WHERE id_usuario = ? 
      AND tipo_permiso = 'Vacaciones' 
      AND estado = 'Pendiente'
) AS has_pending";
$stmtP = $conn->prepare($sql_has_pending);
$stmtP->bind_param('i', $id_usuario);
$stmtP->execute();
$resP = $stmtP->get_result()->fetch_assoc();
$stmtP->close();
$has_pending = (int)($resP['has_pending'] ?? 0);

// 🔵 Suma actual del año SOLO con solicitudes Aprobadas por RRHH
$sql_suma = "SELECT COALESCE(SUM(v.dias), 0) AS total_dias
             FROM vacaciones v
             INNER JOIN solicitudes s ON v.id_solicitud = s.id
             WHERE v.id_usuario = ?
               AND YEAR(v.fecha_inicio) = ?
               AND s.estado = 'Aprobada por Recursos Humanos'";
$stmt_sum = $conn->prepare($sql_suma);
$stmt_sum->bind_param('ii', $id_usuario, $anio);
$stmt_sum->execute();
$stmt_sum->bind_result($total_actual);
$stmt_sum->fetch();
$stmt_sum->close();

// 🔵 NUEVO: Guardia solicitada (“Pendiente” + 12 alcanzados SOLO con aprobadas RRHH)
if ($has_pending && (float)$total_actual >= (float)$LIMITE_VACACIONES_ANUAL) {
    $conn->rollback();
    $_SESSION['error_message'] = "Tienes vacaciones pendientes de aprobación y ya alcanzaste el límite de 12 días aprobados por Recursos Humanos.";
    header("Location: pase_vacaciones.php.php");
    exit;
}

// Guardia previa: no sobrepasar el límite con esta nueva solicitud
$total_posterior = (float)$total_actual + (float)$dias_solicitados;
if ($total_posterior > $LIMITE_VACACIONES_ANUAL) {
    $conn->rollback();
    $_SESSION['error_message'] = "No es posible registrar la solicitud. Llevas acumulados {$total_actual} días aprobados en {$anio}. Solicitar {$dias_solicitados} supera el máximo de {$LIMITE_VACACIONES_ANUAL}.";
    header("Location: pase_vacaciones.php.php");
    exit;
}

    // Insert solicitud
    $sql_solicitudes = "INSERT INTO solicitudes 
        (id_usuario, unidad, puesto, nombre_jefe_inmediato, fecha_solicitud, tipo_permiso, descripcion_motivo, fecha_salida, horario_permiso, firma_funcionario, estado, nombre_funcionario_firma)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    $stmt_solicitudes = $conn->prepare($sql_solicitudes);
    $stmt_solicitudes->bind_param(
        'isssssssssss',
        $id_usuario,
        $unidad,
        $puesto,
        $nombre_jefe_inmediato,
        $fecha_solicitud,
        $tipo_permiso,
        $descripcion_motivo,
        $fecha_salida,
        $dias_solicitados,
        $firma_funcionario,
        $estado,
        $nombre_funcionario_firma
    );
    if (!$stmt_solicitudes->execute()) {
        throw new Exception("Error al guardar la solicitud: " . $stmt_solicitudes->error);
    }
    $solicitud_id = $conn->insert_id;
    $stmt_solicitudes->close();

    // Insert vacaciones ligado a la solicitud
    $observaciones = "Generada desde el pase de Vacaciones, ID de la Solicitud: $solicitud_id";
    $sql_vacaciones = "INSERT INTO vacaciones 
        (id_usuario, fecha_inicio, fecha_fin, dias, observaciones, id_solicitud)
        VALUES (?, ?, ?, ?, ?, ?)";
    $stmt_vac = $conn->prepare($sql_vacaciones);
    $stmt_vac->bind_param("issdsi", $id_usuario, $fecha_salida, $fecha_fin, $dias_solicitados, $observaciones, $solicitud_id);
    if (!$stmt_vac->execute()) {
        throw new Exception("Error al guardar vacaciones: " . $stmt_vac->error);
    }
    $stmt_vac->close();

    $conn->commit();

    // Notificaciones (fuera de la transacción)
    $stmt_u = $conn->prepare("SELECT CorreoElectronico, Nom, Ape, cod_jefe FROM accounts WHERE ID = ?");
    $stmt_u->bind_param('i', $id_usuario);
    $stmt_u->execute();
    $result_usuario = $stmt_u->get_result();
    $solicitante = $result_usuario->fetch_assoc();
    $stmt_u->close();

    if ($solicitante) {
        $to_solicitante = $solicitante['CorreoElectronico'];
        $nombre_solicitante = trim($solicitante['Nom'] . ' ' . $solicitante['Ape']);
        $subject_solicitante = "Nueva Solicitud de Vacaciones Registrada - Estado Pendiente";
        $body_solicitante = "
            <h2>Hola, {$nombre_solicitante}</h2>
            <p>Hemos recibido tu solicitud de vacaciones.</p>
            <p><strong>ID de Solicitud:</strong> {$solicitud_id}</p>
            <p><strong>Fechas:</strong> {$fecha_salida} al {$fecha_fin}</p>
            <p><strong>Días solicitados:</strong> {$dias_solicitados}</p>
            <p>Estado: {$estado}</p>
            <br><p>Saludos,<br>Equipo de RRHH - PROASOL</p>";
        @sendEmail($to_solicitante, $subject_solicitante, $body_solicitante);
    }

    $cod_jefe = $solicitante['cod_jefe'] ?? null;
    if ($cod_jefe) {
        $stmt_j = $conn->prepare("SELECT correo, nombre_jefe FROM jefes WHERE cod_jefe = ?");
        $stmt_j->bind_param('i', $cod_jefe);
        $stmt_j->execute();
        $result_j = $stmt_j->get_result();
        $jefe = $result_j->fetch_assoc();
        $stmt_j->close();

        if ($jefe) {
            $to_jefe = $jefe['correo'];
            $nombre_jefe = $jefe['nombre_jefe'];
            $nombre_solicitante = isset($nombre_solicitante) ? $nombre_solicitante : $nombre_funcionario_firma;
            $subject_jefe = "Nueva Solicitud de Vacaciones Pendiente - Acción Requerida";
            $body_jefe = "
                <h2>Hola, {$nombre_jefe}</h2>
                <p>Se ha creado una nueva solicitud de vacaciones que requiere tu aprobación.</p>
                <p><strong>ID de Solicitud:</strong> {$solicitud_id}</p>
                <p><strong>Solicitante:</strong> {$nombre_solicitante}</p>
                <p><strong>Fechas:</strong> {$fecha_salida} al {$fecha_fin}</p>
                <p><strong>Días solicitados:</strong> {$dias_solicitados}</p>
                <br><p>Saludos,<br>Equipo de RRHH - PROASOL</p>";
            @sendEmail($to_jefe, $subject_jefe, $body_jefe);
        }
    }

    $_SESSION['success_message'] = "✅ Solicitud enviada. Días: {$dias_solicitados}. Acumulado anual: {$total_posterior}/{$LIMITE_VACACIONES_ANUAL}.";
    header("Location: vacaciones.php");
    exit;

} catch (Exception $e) {
    $conn->rollback();
    $_SESSION['error_message'] = "❌ " . $e->getMessage();
    header("Location: pase_vacaciones.php.php");
    exit;
}

$conn->close();
